/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.recommenders.internal.livedoc.javadoc;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.eclipse.recommenders.internal.livedoc.ModelRepoException;
import org.eclipse.recommenders.livedoc.providers.ILivedocProvider;
import org.eclipse.recommenders.livedoc.providers.LiveDocProviderException;
import org.eclipse.recommenders.livedoc.providers.ProviderOutput;
import org.eclipse.recommenders.livedoc.utils.HtmlUtils;
import org.eclipse.recommenders.livedoc.utils.HtmlUtils.HtmlTable;
import org.eclipse.recommenders.livedoc.utils.LiveDocUtils;
import org.eclipse.recommenders.models.IModelIndex;
import org.eclipse.recommenders.models.IModelRepository;
import org.eclipse.recommenders.models.ModelIndex;
import org.eclipse.recommenders.models.ModelRepository;
import org.eclipse.recommenders.models.ProjectCoordinate;
import org.eclipse.recommenders.utils.Urls;
import org.eclipse.recommenders.utils.Zips;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.sun.javadoc.AnnotationTypeDoc;
import com.sun.javadoc.AnnotationTypeElementDoc;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.ConstructorDoc;
import com.sun.javadoc.Doc;
import com.sun.javadoc.FieldDoc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.PackageDoc;
import com.sun.javadoc.RootDoc;
import com.sun.tools.doclets.formats.html.markup.RawHtml;
import com.sun.tools.doclets.internal.toolkit.Content;
import com.sun.tools.doclets.internal.toolkit.taglets.BaseTaglet;
import com.sun.tools.doclets.internal.toolkit.taglets.TagletWriter;

public class RecommendersTaglet extends BaseTaglet {

    private static final String NAME = "recommenders";
    private static final String XMARK = "&#x2717;";
    private static final File TEMP_DIR = new File(FileUtils.getTempDirectory(), "livedoc");
    private static final File MODELSREPO_CACHE_DIR = new File(TEMP_DIR, "modelsRepo/cache");
    private static final File MODELSREPO_INDEX_DIR = new File(TEMP_DIR, "modelsRepo/indexes");
    private static final Logger LOG = LoggerFactory.getLogger(RecommendersTaglet.class);

    private final ProjectCoordinate projectCoordinate;
    private final List<URL> modelRepositoryURLs;
    private final List<ILivedocProvider<?>> providers;
    private final boolean highlight;

    private IModelRepository compositeModelRepository;
    private IModelIndex compositeModelIndex;

    private PackageDoc currentPackageDoc;
    private ClassDoc currentClassDoc;
    private AnnotationTypeDoc currentAnnotationTypeDoc;

    public RecommendersTaglet(ProjectCoordinate pc, List<URL> modelRepositoryURLs, List<ILivedocProvider<?>> providers,
            boolean highlight) {
        this.projectCoordinate = pc;
        this.modelRepositoryURLs = modelRepositoryURLs;
        this.providers = providers;
        this.highlight = highlight;
    }

    @Override
    public String getName() {
        return NAME;
    }

    public void setUp() throws RecommendersTagletException {
        try {
            prepareModelIndexAndRepo();
        } catch (ModelRepoException e) {
            // TODO: Should taglets be also executed when no repo exists?
            LOG.error("Couldn't prepare model repository, skipping Livedoc providers.", e);
            throw new RecommendersTagletException(e);
        }

        Iterator<ILivedocProvider<?>> it = providers.iterator();
        while (it.hasNext()) {
            ILivedocProvider<?> provider = it.next();

            try {
                provider.setUp(projectCoordinate, compositeModelRepository, compositeModelIndex);
                provider.getConfiguration().setHighlight(highlight);
            } catch (LiveDocProviderException e) {
                it.remove();
                LOG.error("Provider \"{}\" couldn't be set up, skipping it.", provider.getClass().getSimpleName(), e);
            }
        }
    }

    private void prepareModelIndexAndRepo() throws ModelRepoException {

        MODELSREPO_CACHE_DIR.mkdirs();
        MODELSREPO_INDEX_DIR.mkdirs();

        List<IModelRepository> componentModelRepositories = Lists.newArrayList();
        List<IModelIndex> componentModelIndexes = Lists.newArrayList();

        for (URL url : modelRepositoryURLs) {
            File repo = new File(MODELSREPO_CACHE_DIR, Urls.mangle(url.toString()));
            try {
                IModelRepository modelRepository = new ModelRepository(repo, url.toExternalForm());
                componentModelRepositories.add(modelRepository);

                Optional<File> zippedIndex = modelRepository.resolve(IModelIndex.INDEX, false);
                if (zippedIndex.isPresent()) {
                    IModelIndex index = null;
                    try {
                        File indexDir = new File(MODELSREPO_INDEX_DIR, Urls.mangle(url.toString()));
                        indexDir.mkdir();
                        Zips.unzip(zippedIndex.get(), indexDir);
                        index = new ModelIndex(indexDir);
                        index.open();
                        componentModelIndexes.add(index);
                    } catch (IOException e) {
                        LOG.warn("No Code Recommenders index available for repository \"{}\".", url, e);
                    }
                }
            } catch (Exception e) {
                throw new ModelRepoException(e);
            }
        }

        compositeModelRepository = new CompositeModelRepository(componentModelRepositories);
        compositeModelIndex = new CompositeModelIndex(componentModelIndexes);
    }

    @Override
    public Content getTagletOutput(Doc holder, TagletWriter writer) throws IllegalArgumentException {

        // Overview
        if (holder instanceof RootDoc) {
            return aggreagteOverviewDoc((RootDoc) holder, writer);
        }

        // Packages
        if (holder instanceof PackageDoc) {
            return aggregatePackageDoc((PackageDoc) holder, writer);
        }

        // Class and Interfaces
        if (classOrInterface(holder)) {
            return aggregateClassDoc((ClassDoc) holder, writer);
        }

        // Fields
        if (holder.isField()) {
            return aggregateFieldDoc((FieldDoc) holder, writer);
        }

        // Constructors
        if (holder.isConstructor()) {
            return aggregateConstructorDoc((ConstructorDoc) holder, writer);
        }

        // Methods
        if (holder.isMethod()) {
            return aggregateMethodDoc((MethodDoc) holder, writer);
        }

        // Annotation Types
        if (holder.isAnnotationType()) {
            return aggregateAnnotationTypeDoc((AnnotationTypeDoc) holder, writer);
        }

        // Annotation Type Elements
        if (holder.isAnnotationTypeElement()) {
            return aggregateAnnotationTypeElementDoc((AnnotationTypeElementDoc) holder, writer);
        }

        return null;
    }

    private Content aggreagteOverviewDoc(RootDoc holder, TagletWriter writer) {
        
        return output(writer, callProviders((provider) -> provider.documentOverview(holder)));
    }

    private Content aggregatePackageDoc(PackageDoc holder, TagletWriter writer) {
        newHolderLifecycle(holder, this.currentPackageDoc, new IDocLifecycle<PackageDoc>() {

            @Override
            public void endDoc(PackageDoc oldDoc, ILivedocProvider<?> provider) {
                provider.endPackage(oldDoc);
            }

            @Override
            public void beginDoc(PackageDoc newDoc, ILivedocProvider<?> provider) {
                provider.beginPackage(newDoc);
            }
        });
        this.currentPackageDoc = holder;

        // Aggregate ClassDoc Output for pretty print package level presentation

        HtmlTable table = new HtmlTable("Code Recommenders Summary", "typeSummary",
                "Code Recommenders Summary table, listing Code Recommendations for Types");

        table.addColumn("Class");

        ClassDoc[] classDocs = holder.allClasses();

        // Create a copy of providers list (as we delete the analytics provider)
        List<ILivedocProvider<?>> providers = Lists.newArrayList(this.providers);
        Iterables.removeIf(providers, new Predicate<ILivedocProvider<?>>() {

            @Override
            public boolean apply(ILivedocProvider<?> provider) {
                return provider.getId().equals("analytics");
            }
        });

        Arrays.sort(classDocs);
        for (int i = 0; i < classDocs.length; i++) {
            ClassDoc classDoc = classDocs[i];
            String[] row = new String[providers.size() + 1];
            row[0] = LiveDocUtils.htmlLink(classDoc).toString();

            int y = 1;
            // Does at least one provider have recommendations for this classdoc?
            boolean recommendations = false;

            for (Iterator<ILivedocProvider<?>> iterator = providers.iterator(); iterator.hasNext();) {
                ILivedocProvider<?> provider = (ILivedocProvider<?>) iterator.next();

                // add Provider to table
                if (i == 0) {
                    table.addColumn(provider.getConfiguration().getName());
                }
                provider.beginClass(classDoc);
                ProviderOutput output = provider.documentClass(classDoc);

                // highest possible level, so try this one first
                if (output != null) {
                    row[y] = prettyPrintNumber(output.getNumberOfRecommendations());
                    recommendations = true;
                } else {
                    // try method level instead
                    int count = 0;
                    for (MethodDoc method : classDoc.methods()) {
                        output = provider.documentMethod(method);

                        // Output for a method? So count + 1
                        if (output != null) {
                            count++;
                        }
                    }
                    if (count != 0) {
                        row[y] = prettyPrintNumber(count);
                        recommendations = true;
                    } else {
                        row[y] = HtmlUtils.grey(XMARK);
                    }
                }
                provider.endClass(classDoc);
                y++;
            }
            if (recommendations) {
                table.addRow(row);
                recommendations = false;
            }
        }

        // get actual ProviderOutputs
        StringBuilder sb = new StringBuilder(table.toString());
        sb.append(callProviders((provider) -> provider.documentPackage(holder)));
        return output(writer, sb.toString());
    }

    private String prettyPrintNumber(int number) {
        StringBuilder sb = new StringBuilder(Integer.toString(number));
        return LiveDocUtils.strong(sb).toString();
    }

    private Content aggregateClassDoc(ClassDoc holder, TagletWriter writer) {

        newHolderLifecycle(holder, this.currentClassDoc, new IDocLifecycle<ClassDoc>() {

            @Override
            public void beginDoc(ClassDoc newDoc, ILivedocProvider<?> provider) {
                provider.beginClass(newDoc);
            }

            @Override
            public void endDoc(ClassDoc oldDoc, ILivedocProvider<?> provider) {
                provider.endClass(oldDoc);
            }
        });

        this.currentClassDoc = holder;

        return output(writer, callProviders((provider) -> provider.documentClass(holder)));
    }

    private Content aggregateFieldDoc(FieldDoc holder, TagletWriter writer) {

        return output(writer, callProviders((provider) -> provider.documentField(holder)));
    }

    private Content aggregateConstructorDoc(ConstructorDoc holder, TagletWriter writer) {

        return output(writer, callProviders((provider) -> provider.documentConstructor(holder)));
    }

    private Content aggregateMethodDoc(MethodDoc holder, TagletWriter writer) {

        return output(writer, callProviders((provider) -> provider.documentMethod(holder)));
    }

    private Content aggregateAnnotationTypeDoc(AnnotationTypeDoc holder, TagletWriter writer) {

        newHolderLifecycle(holder, this.currentAnnotationTypeDoc, new IDocLifecycle<AnnotationTypeDoc>() {

            @Override
            public void beginDoc(AnnotationTypeDoc newDoc, ILivedocProvider<?> provider) {
                provider.beginAnnotationType(newDoc);
            }

            @Override
            public void endDoc(AnnotationTypeDoc oldDoc, ILivedocProvider<?> provider) {
                provider.endAnnotationType(oldDoc);
            }
        });

        this.currentAnnotationTypeDoc = holder;

        return output(writer, callProviders((provider) -> provider.documentAnnotationType(holder)));
    }

    private Content aggregateAnnotationTypeElementDoc(AnnotationTypeElementDoc holder, TagletWriter writer) {
        return output(writer, callProviders((provider) -> provider.documentAnnotationTypeElement(holder)));
    }

    public void tearDown() throws RecommendersTagletException {

        // close open packages, Classes, AnnotationTypes
        // tearDown

        providers.forEach(provider -> {
            try {
                provider.endClass(currentClassDoc);
                provider.endAnnotationType(currentAnnotationTypeDoc);
                provider.endPackage(currentPackageDoc);
                provider.tearDown();
            } catch (LiveDocProviderException e) {
                LOG.error("Couldn't tear down Provider \"{}\"", provider.getId(), e);
            }
        });
    }

    private boolean classOrInterface(Doc holder) {
        return (holder.isClass() || holder.isInterface());
    }

    private <T extends Doc> void newHolderLifecycle(T newHolder, T currentHolder, IDocLifecycle<T> lifecycle) {

        if (holderChanged(currentHolder, newHolder)) {
            
            providers.forEach(provider -> {
                
                // first case is special, no PackageDoc there
                if (currentHolder != null) {
                    lifecycle.endDoc(currentHolder, provider);
                }
                lifecycle.beginDoc(newHolder, provider);
            });
        }
    }

    private boolean holderChanged(Doc currentHolder, Doc newHolder) {
        return !newHolder.equals(currentHolder);
    }

    private Content output(TagletWriter writer, String output) {
        if (output.length() == 0) {
            return null;
        }

        Content content = writer.getOutputInstance();
        RawHtml html = new RawHtml(output);
        content.addContent(html);
        return content;
    }

    private String callProviders(IProviderCall call) {
        StringBuilder sb = new StringBuilder();

        providers.forEach(provider -> {
            
            ProviderOutput output = call.document(provider);
            if (output != null) {
                sb.append(output.getHtmlCode());
            }
        });

        return sb.toString();
    }

    private interface IDocLifecycle<T extends Doc> {

        void beginDoc(T newDoc, ILivedocProvider<?> provider);

        void endDoc(T oldDoc, ILivedocProvider<?> provider);

    }

    private interface IProviderCall {
        ProviderOutput document(ILivedocProvider<?> provider);
    }
}
