/**
 * Copyright (c) 2013 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick Gottschaemmer - initial API and implementation.
 */
package org.eclipse.recommenders.livedoc.utils;

import static com.google.common.collect.ComparisonChain.start;
import static org.eclipse.recommenders.utils.Pair.newPair;
import static org.eclipse.recommenders.utils.names.Names.src2vmMethod;
import static org.eclipse.recommenders.utils.names.Names.src2vmType;
import static org.eclipse.recommenders.utils.names.Names.vm2srcQualifiedType;
import static org.eclipse.recommenders.utils.names.Names.vm2srcSimpleTypeName;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.eclipse.recommenders.livedoc.Livedoc;
import org.eclipse.recommenders.utils.Pair;
import org.eclipse.recommenders.utils.Recommendation;
import org.eclipse.recommenders.utils.Recommendations;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.IPackageName;
import org.eclipse.recommenders.utils.names.ITypeName;
import org.eclipse.recommenders.utils.names.Names;
import org.eclipse.recommenders.utils.names.VmMethodName;
import org.eclipse.recommenders.utils.names.VmTypeName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.Beta;
import com.google.common.base.Optional;
import com.google.common.collect.Maps;
import com.google.common.collect.Ordering;
import com.google.common.io.Files;
import com.sun.javadoc.ClassDoc;
import com.sun.javadoc.Doc;
import com.sun.javadoc.MethodDoc;
import com.sun.javadoc.PackageDoc;
import com.sun.javadoc.Parameter;
import com.sun.javadoc.Type;

public class LiveDocUtils {

    public static final String BR = "<br>";
    public static final String CODE = "<code>";
    public static final String CODE_END = "</code>";
    public static final String DD = "<dd>";
    public static final String DD_END = "</dd>";
    public static final String DT = "<dt>";
    public static final String DT_END = "</dt>";
    public static final String LI = "<li>";
    public static final String LI_END = "</li>";
    public static final String STRONG_END = "</strong>";
    public static final String STRONG = "<strong>";
    public static final String UL = "<ul>";
    public static final String UL_END = "</ul>";

    private static Map<Pair<ClassDoc, IMethodName>, ClassDoc> methodHolderCache = Maps.newHashMapWithExpectedSize(6000);

    private static Comparator<Recommendation<IMethodName>> C_BY_RELEVANCE_THEN_NAME = new Comparator<Recommendation<IMethodName>>() {

        @Override
        public int compare(final Recommendation<IMethodName> o1, final Recommendation<IMethodName> o2) {
            return start().compare(Recommendations.asPercentage(o1), Recommendations.asPercentage(o2))
                    .compare(o2.getProposal().getName(), o1.getProposal().getName()).result();
        }
    };

    public static <R extends Recommendation<IMethodName>> List<R> topMethods(final Iterable<R> recommendations,
            final int numberOfTopElements, final double minRelevance) {
        return Ordering.from(C_BY_RELEVANCE_THEN_NAME).greatestOf(
                Recommendations.filterRelevance(recommendations, minRelevance), numberOfTopElements);
    }

    public static String htmlMethodSignature(IMethodName method) {

        StringBuilder sb = new StringBuilder();
        sb.append(CODE);
        sb.append(simpleMethodName(method));
        sb.append("(");

        for (int i = 0; i < method.getParameterTypes().length; i++) {

            ITypeName parameter = method.getParameterTypes()[i];
            sb.append(vm2srcQualifiedType(parameter));
            if (i < method.getParameterTypes().length - 1) {
                sb.append(", ");
            }
        }

        sb.append(")");
        sb.append(CODE_END);

        return sb.toString();
    }

    private static String simpleMethodName(IMethodName method) {
        return method.isInit() ? vm2srcSimpleTypeName(method.getDeclaringType()) : method.getName();
    }

    /**
     * Doesn't work with arrays!
     * 
     */
    @Deprecated
    public static IMethodName asIMethodName(MethodDoc methodDoc) {
        String srcDeclaringType = StringUtils.substringBeforeLast(methodDoc.qualifiedName(), ".");
        String methodName = StringUtils.substringAfterLast(methodDoc.qualifiedName(), ".");
        String[] parameters = asStringArray(methodDoc.parameters());

        return VmMethodName.get(src2vmMethod(srcDeclaringType, methodName, parameters, methodDoc.returnType()
                .qualifiedTypeName()));
    }

    public static String[] asStringArray(Parameter[] parameters) {

        String[] result = new String[parameters.length];

        for (int i = 0; i < parameters.length; i++) {
            Parameter parameter = parameters[i];
            result[i] = parameter.type().qualifiedTypeName();
        }
        return result;
    }

    public static ITypeName extractTypeName(Doc holder) {
        if (holder.isMethod()) {
            String typeName = StringUtils.substringBefore(holder.toString(), "(");
            typeName = StringUtils.substringBeforeLast(typeName, ".");
            return VmTypeName.get(src2vmType(typeName));
        } else {
            return VmTypeName.get(src2vmType(holder.toString()));
        }
    }

    public static URI relativeUri(ITypeName type) throws URISyntaxException {
        StringBuilder sb = new StringBuilder();
        sb.append(climbUp(type.getPackage()))
                .append(type.getPackage().getIdentifier())
                .append("/")
                .append(type.getClassName())
                .append(".html");
        return new URI(sb.toString());
    }

    public static URI relativeUri(ClassDoc classDoc) throws URISyntaxException {
        StringBuilder sb = new StringBuilder();
        sb.append(climbUp(classDoc.containingPackage()))
                .append(StringUtils.replace(classDoc.containingPackage().name(), ".", "/"))
                .append("/")
                .append(classDoc.name())
                .append(".html");
        return new URI(sb.toString());
    }

    public static URI relativeUri(IMethodName method) throws URISyntaxException {
        StringBuilder fragment = new StringBuilder(simpleMethodName(method));
        fragment.append(listParameterTypes(method));
        return new URI(null, relativeUri(method.getDeclaringType()).toString(), fragment.toString());
    }

    private static String climbUp(IPackageName packageName) {
        int length = StringUtils.countMatches(packageName.getIdentifier(), "/");
        String result = "";
        for (int i = 0; i <= length; i++) {
            result += "../";
        }
        return result;
    }

    private static String climbUp(PackageDoc packageDoc) {
        int length = StringUtils.countMatches(packageDoc.name(), ".");
        String result = "";
        for (int i = 0; i <= length; i++) {
            result += "../";
        }
        return result;
    }

    public static StringBuilder htmlLink(IMethodName method) {

        StringBuilder sb = new StringBuilder("<a href=");
        // create relative URL
        sb.append("\"");
        try {
            sb.append(relativeUri(method).toString());
        } catch (URISyntaxException e) {
            System.err.println("Couldn't create HTML Link for:" + method.getIdentifier());
            e.printStackTrace();
        }

        // title and appearance
        sb.append("\">").append(simpleMethodName(method));
        sb.append("</a>");
        return sb;
    }

    // TODO: refactor this one, so htmlLink(ClassDoc, IMethodName) is unnecessary, use the
    // simple htmlLink(IMethodName) instead
    @Beta
    public static StringBuilder htmlLink(ClassDoc callee, IMethodName methodName) {

        Pair<ClassDoc, IMethodName> pair = newPair(callee, methodName);

        // look up cache, maybe we know this one already
        if (methodHolderCache.containsKey(pair)) {
            ITypeName typeName = extractTypeName(methodHolderCache.get(pair));
            return htmlLink(typeName, methodName);
        }

        Optional<ClassDoc> holder = searchMethodHolder(callee, methodName);

        if (holder.isPresent()) {
            ClassDoc declaringClass = holder.get();
            methodHolderCache.put(pair, declaringClass);
            ITypeName typeName = extractTypeName(declaringClass);
            return htmlLink(typeName, methodName);
        }

        return new StringBuilder(methodName.getName());
    }

    private static Optional<ClassDoc> searchMethodHolder(ClassDoc callee, IMethodName methodName) {

        // we reached the root of the hierarchy, so return absent
        if (callee == null) {
            return Optional.absent();
        }

        // look for method in ClassDoc
        if (methodInClassDoc(callee, methodName)) {
            return Optional.of(callee);
        }

        // search in superclasses, maybe later in interfaces top-down
        Optional<ClassDoc> result = searchMethodHolder(callee.superclass(), methodName);

        // if result is present AND we climp up the hierarchy, then result can't be an interface,
        // so return. otherwise, keep searching till bottom end of recursion.
        if (result.isPresent() && !result.get().isInterface()) {
            return result;
        }

        // search in interfaces, return new result or old result if absent
        return searchInterfaces(callee, methodName).or(result);
    }

    private static Optional<ClassDoc> searchInterfaces(ClassDoc callee, IMethodName methodName) {

        // search in direct interfaces
        for (ClassDoc interfaze : callee.interfaces()) {
            if (methodInClassDoc(interfaze, methodName)) {
                return Optional.of(interfaze);
            }
        }

        // nothing found? search in super interfaces
        for (ClassDoc interfaze : callee.interfaces()) {
            return searchInterfaces(interfaze, methodName);
        }

        return Optional.absent();
    }

    private static boolean methodInClassDoc(ClassDoc callee, IMethodName methodName) {

        MethodDoc[] methods = callee.methods();
        for (MethodDoc method : methods) {
            if (returnTypeEquals(methodName, method) && simpleNameEquals(methodName, method)
                    && parameterTypesEquals(methodName, method)) {
                return true;
            }
        }
        return false;
    }

    public static StringBuilder htmlLink(ITypeName declaringClass, IMethodName methodName) {
        methodName = VmMethodName.get(declaringClass.getIdentifier(), methodName.getSignature());
        return htmlLink(methodName);
    }

    public static StringBuilder htmlLink(ClassDoc classDoc) {

        StringBuilder sb = new StringBuilder("<a href=");

        // create relative URL
        sb.append("\"");
        try {
            sb.append(relativeUri(classDoc).toString());
        } catch (URISyntaxException e) {
            System.err.println("Couldn't create HTML Link for:" + classDoc.qualifiedName());
            e.printStackTrace();
        }
        sb.append("\"");

        // title and appearance
        sb.append(" title=\"class in ")
                .append(classDoc.containingPackage().name())
                .append("\">")
                .append(classDoc.name())
                .append("</a>");
        return sb;
    }

    public static StringBuilder htmlLink(ITypeName type) {

        StringBuilder sb = new StringBuilder("<a href=");

        // create relative URL
        sb.append("\"");
        try {
            sb.append(relativeUri(type).toString());
        } catch (URISyntaxException e) {
            System.err.println("Couldn't create HTML Link for:" + type.getIdentifier());
            e.printStackTrace();
        }
        sb.append("\"");

        // title and appearance
        sb.append(" title=\"class in ").append(StringUtils.replace(type.getPackage().getIdentifier(), "/", "."))
                .append("\">").append(type.getClassName()).append("</a>");
        return sb;
    }

    public static boolean methodSignatureEquals(IMethodName methodName, MethodDoc methodDoc) {

        if (!qualifiedNameEquals(methodName, methodDoc)) {
            return false;
        }
        if (!parameterTypesEquals(methodName, methodDoc)) {
            return false;
        }
        if (!returnTypeEquals(methodName, methodDoc)) {
            return false;
        }
        return true;
    }

    private static boolean returnTypeEquals(IMethodName methodName, MethodDoc methodDoc) {
        return typeEquals(methodName.getReturnType(), methodDoc.returnType());
    }

    private static boolean parameterTypesEquals(IMethodName methodName, MethodDoc methodDoc) {

        ITypeName[] methodNameParameters = methodName.getParameterTypes();

        Parameter[] methodDocParameters = methodDoc.parameters();

        if (methodNameParameters.length == methodDocParameters.length) {
            for (int i = 0; i < methodNameParameters.length; i++) {
                if (!typeEquals(methodNameParameters[i], methodDocParameters[i].type()))
                    return false;
            }
            return true;

        } else {
            return false;
        }
    }

    /**
     * @param typeName
     *            {@link ITypeName}e from o.e.r.
     * @param type
     *            {@link Type} or {@link ClassDoc} from c.s.j., as {@link ClassDoc} extends {@link Type}
     */
    public static boolean typeEquals(ITypeName typeName, Type type) {

        if (Names.vm2srcQualifiedType(typeName).equals(removeGenerics(type))) {
            return true;
        }

        // Maybe its a generic?
        // TODO: Very dirty & unreliable!
        if (type.asTypeVariable() != null) {
            Type[] bounds = type.asTypeVariable().bounds();

            // bound could/should implicit be Object!
            if (bounds.length == 0) {
                if (typeName.isArrayType()) {
                    boolean dimensionsEqual = StringUtils.countMatches(type.dimension(), "[]") == typeName
                            .getArrayDimensions();

                    return dimensionsEqual && typeName.getArrayBaseType().equals(VmTypeName.OBJECT);
                }
                return typeName.equals(VmTypeName.OBJECT);
            }

            // typeName has to be a subtype of bounds for equality/matching
            // TODO: checksubclassOf(...)
        }
        return false;
    }

    private static String removeGenerics(Type type) {
        String generics = StringUtils.defaultString(StringUtils.substringBetween(type.toString(), "<", ">"));
        String toRemove = "<".concat(generics).concat(">");
        return StringUtils.remove(type.toString(), toRemove);
    }

    /**
     * @return true, if declaring types and simple method names are equal
     */
    private static boolean qualifiedNameEquals(IMethodName methodName, MethodDoc methodDoc) {

        if (!declaringTypeMatches(methodName, methodDoc)) {
            return false;
        }
        if (!simpleNameEquals(methodName, methodDoc)) {
            return false;
        }
        return true;
    }

    private static boolean declaringTypeMatches(IMethodName methodName, MethodDoc methodDoc) {
        return typeEquals(methodName.getDeclaringType(), methodDoc.containingClass());
    }

    private static boolean simpleNameEquals(IMethodName methodName, MethodDoc methodDoc) {
        return methodName.getName().equals(methodDoc.name());
    }

    public static String listParameterTypes(IMethodName method) {
        StringBuilder sb = new StringBuilder();
        sb.append("(");

        ITypeName[] parameterTypes = method.getParameterTypes();
        for (int i = 0; i < parameterTypes.length; i++) {

            sb.append(vm2srcQualifiedType(parameterTypes[i]));
            if (i < parameterTypes.length - 1) {
                sb.append(", ");
            }
        }
        sb.append(")");
        return sb.toString();
    }

    public static StringBuilder strong(StringBuilder textToStrong) {
        textToStrong.insert(0, STRONG);
        textToStrong.append(STRONG_END);
        return textToStrong;
    }

    public static String strong(String textToStrong) {
        return strong(new StringBuilder(textToStrong)).toString();
    }

    public static StringBuilder size(String size, StringBuilder textToSize) {
        textToSize.insert(0, "<span style=\"font-size:" + size + "\">").append("</span>");
        return textToSize;
    }

    public static String size(String size, String textToSize) {
        return size(size, new StringBuilder(textToSize)).toString();
    }

    public static StringBuilder code(StringBuilder textToCode) {
        textToCode.insert(0, CODE).append(CODE_END);
        return textToCode;
    }

    public static String code(String textToCode) {
        return code(new StringBuilder(textToCode)).toString();
    }

    public static String surroundWith(String begin, String text, String end) {
        return new StringBuilder(begin).append(text).append(end).toString();
    }

    /**
     * 
     * @param color
     *            html color code, e.g. #0000FF
     */
    public static String color(String color, String text) {
        return surroundWith(surroundWith("<font color=\"", color, "\">"), text, "</font>");
    };

    public static String highlight(String string) {
        return color("#0000FF", string);
    }
}
