/**
 * Copyright (c) 2014 Patrick Gottschaemmer.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Patrick Gottschaemmer - initial API and implementation.
 */
package org.eclipse.recommenders.livedoc.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.LineNumberReader;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.JavaVersion;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Files;

public class LivedocChecks {

    private static final String ENV_JAVA_HOME = System.getenv("JAVA_HOME");
    private static final Logger LOG = LoggerFactory.getLogger(LivedocChecks.class);

    public static void preconditions(boolean printToConsole) throws LivedocPreconditionException {
        checkJavaVersion(printToConsole);
        checkToolsJar(printToConsole);
    }

    private static void checkJavaVersion(boolean printToConsole) throws LivedocPreconditionException {
        if (!SystemUtils.isJavaVersionAtLeast(JavaVersion.JAVA_1_8)) {
            LivedocPreconditionException e = new LivedocPreconditionException("Wrong Java Version");
            String error = "Livedoc is only compatible with Java 1.8";
            LOG.error(error, e);
            if (printToConsole) {
                System.err.println(e.getMessage());
                System.err.println(error);
            }
            throw e;
        }
    }

    private static void checkToolsJar(boolean printToConsole) throws LivedocPreconditionException {

        try {
            // Is tools.jar present and loaded?
            Class.forName("com.sun.tools.doclets.internal.toolkit.Content");

            // Does java.home point to the right JDK (not JRE only!) version?
            File release = new File(SystemUtils.JAVA_HOME, "../release");

            // If it's a JDK, a tools.jar should be there (and therefore, successfully loaded by o.e.r.3rd.tools_jar)
            File toolsJar = new File(SystemUtils.JAVA_HOME, "../lib/tools.jar");

            if (toolsJar.isFile() && correctJavaVersionInReleaseFile(release)) {
                return;
            }

            // Or at least in environment variable $JAVA_HOME?
            release = new File(ENV_JAVA_HOME, "release");
            toolsJar = new File(ENV_JAVA_HOME, "lib/tools.jar");

            if (toolsJar.isFile() && correctJavaVersionInReleaseFile(release)) {
                return;
            }

            // Still no? Bad luck then...
            throw new Exception();

        } catch (Exception e) {
            String error = String
                    .format("Livedoc could not find a correct JDK8 tools.jar, neither in \"%s\" (java.home) nor in \"%s\" ($JAVA_HOME)",
                            SystemUtils.JAVA_HOME, ENV_JAVA_HOME);
            LOG.error(error, e);
            if (printToConsole) {
                System.err.println(error);
            }
            throw new LivedocPreconditionException(e);
        }
    }

    private static boolean correctJavaVersionInReleaseFile(File release) throws IOException {
        Pattern regexp = Pattern.compile("JAVA\\_VERSION=\"1\\.8\\..*");

        Matcher matcher = regexp.matcher("");

        BufferedReader reader = Files.newReader(release, StandardCharsets.UTF_8);
        LineNumberReader lineReader = new LineNumberReader(reader);

        String line = null;
        while ((line = lineReader.readLine()) != null) {
            matcher.reset(line);
            if (matcher.find()) {
                return true;
            }
        }
        return false;
    }

}
