/*******************************************************************************
 * Copyright (c) 2013 Stefan Prisca.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Stefan Prisca - initial API and implementation
 ******************************************************************************/
package org.eclipse.recommenders.templates.rcp.syntaxhighlighting;

import org.eclipse.jdt.internal.ui.javaeditor.SemanticHighlighting;
import org.eclipse.jdt.internal.ui.javaeditor.SemanticHighlightings;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.jdt.ui.text.IJavaColorConstants;
import org.eclipse.jdt.ui.text.JavaTextTools;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.xtext.ui.editor.syntaxcoloring.DefaultHighlightingConfiguration;
import org.eclipse.xtext.ui.editor.syntaxcoloring.IHighlightingConfigurationAcceptor;
import org.eclipse.xtext.ui.editor.utils.TextStyle;

@SuppressWarnings("restriction")
public class TemplatesHighlightingConfiguration extends DefaultHighlightingConfiguration {

    public static final String TEMPLATE_IDENTIFIER_ID = "templateIdentifier";
    public static final String TEMPLATE_KEYWORD_ID = "templateKeyword";
    public static final String TEMPLATE_SPECIAL_TYPE_ID = "templateSpecialType";
    public static final String JAVA_IDENTIFIER_ID = "javaIdentifier";
    public static final String JAVA_MULTI_LINE_COMMENT_ID = "javaMultiLineComment";
    public static final String JAVA_SINGLE_LINE_COMMENT_ID = "javaSingleLineComment";
    public static final String JAVA_BRACKETS_ID = "javaBrackets";
    public static final String JAVA_OPERATOR_ID = "javaOperator";
    public static final String JAVA_RETURN_KEYWORD_ID = "javaReturnKeyword";

    private final IPreferenceStore jdtPrefs;
    private final IColorManager colorMan;

    public TemplatesHighlightingConfiguration() {
        this.jdtPrefs = PreferenceConstants.getPreferenceStore();
        this.colorMan = new JavaTextTools(jdtPrefs).getColorManager();
    }

    @Override
    public void configure(IHighlightingConfigurationAcceptor acceptor) {
        acceptor.acceptDefaultHighlighting(TEMPLATE_IDENTIFIER_ID, "Template Identifier", templateIdentifierTextStyle());
        acceptor.acceptDefaultHighlighting(TEMPLATE_KEYWORD_ID, "Template Keyword", templateKeywordTextStyle());
        acceptor.acceptDefaultHighlighting(TEMPLATE_SPECIAL_TYPE_ID, "Template Special Type",
                templateSpecialTypeTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_IDENTIFIER_ID, "Java Identifier", defaultTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_MULTI_LINE_COMMENT_ID, "Multi Line Comment",
                javaMultiLineCommentTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_SINGLE_LINE_COMMENT_ID, "Single Line Comment",
                javaSingleLineCommentTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_BRACKETS_ID, "Brackets", javaBracketsTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_OPERATOR_ID, "Operator", javaOperatorTextStyle());
        acceptor.acceptDefaultHighlighting(JAVA_RETURN_KEYWORD_ID, "Keyword 'return'", javaReturnKeywordTextStyle());
        acceptor.acceptDefaultHighlighting(KEYWORD_ID, "Keywords excluding 'return'", keywordTextStyle());
        acceptor.acceptDefaultHighlighting(TASK_ID, "Task Tag", taskTextStyle());
        acceptor.acceptDefaultHighlighting(STRING_ID, "String", stringTextStyle());
        acceptor.acceptDefaultHighlighting(NUMBER_ID, "Number", numberTextStyle());
        acceptor.acceptDefaultHighlighting(DEFAULT_ID, "Default", defaultTextStyle());
        acceptor.acceptDefaultHighlighting(INVALID_TOKEN_ID, "Invalid Symbol", errorTextStyle());
    }

    protected TextStyle templateSpecialTypeTextStyle() {
        TextStyle style = defaultTextStyle().copy();
        style.setColor(new RGB(155, 0, 0));
        style.setStyle(SWT.ITALIC);
        return style;
    }

    protected TextStyle templateIdentifierTextStyle() {
        TextStyle style = defaultTextStyle().copy();
        style.setColor(new RGB(255, 128, 64));
        return style;
    }

    protected TextStyle templateKeywordTextStyle() {
        TextStyle style = defaultTextStyle().copy();
        style.setColor(new RGB(64, 0, 128));
        style.setStyle(SWT.BOLD);
        return style;
    }

    protected TextStyle javaMultiLineCommentTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.commentTextStyle();
        }

        TextStyle mlCommentTextStyle = defaultTextStyle();
        mlCommentTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_MULTI_LINE_COMMENT_BOLD,
                PreferenceConstants.EDITOR_MULTI_LINE_COMMENT_ITALIC,
                PreferenceConstants.EDITOR_MULTI_LINE_COMMENT_UNDERLINE));

        mlCommentTextStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_MULTI_LINE_COMMENT).getRGB());
        return mlCommentTextStyle;
    }

    protected TextStyle javaSingleLineCommentTextStyle() {
        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.commentTextStyle();
        }

        TextStyle slCommentTextStyle = defaultTextStyle();
        slCommentTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_SINGLE_LINE_COMMENT_BOLD,
                PreferenceConstants.EDITOR_SINGLE_LINE_COMMENT_ITALIC,
                PreferenceConstants.EDITOR_SINGLE_LINE_COMMENT_UNDERLINE));

        slCommentTextStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_SINGLE_LINE_COMMENT).getRGB());
        return slCommentTextStyle;
    }

    protected TextStyle javaOperatorTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.defaultTextStyle();
        }

        TextStyle operatorTextStyle = defaultTextStyle();
        operatorTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_JAVA_OPERATOR_BOLD,
                PreferenceConstants.EDITOR_JAVA_OPERATOR_ITALIC, PreferenceConstants.EDITOR_JAVA_OPERATOR_UNDERLINE));

        operatorTextStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_OPERATOR).getRGB());
        return operatorTextStyle;
    }

    protected TextStyle javaBracketsTextStyle() {
        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.defaultTextStyle();
        }

        TextStyle bracketsTextStyle = defaultTextStyle();
        bracketsTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_JAVA_BRACKET_BOLD,
                PreferenceConstants.EDITOR_JAVA_BRACKET_ITALIC, PreferenceConstants.EDITOR_JAVA_BRACKET_UNDERLINE));

        bracketsTextStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_BRACKET).getRGB());
        return bracketsTextStyle;
    }

    protected TextStyle javaReturnKeywordTextStyle() {
        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.defaultTextStyle();
        }

        TextStyle returnKeywordTextStyle = defaultTextStyle();
        returnKeywordTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_JAVA_KEYWORD_RETURN_BOLD,
                PreferenceConstants.EDITOR_JAVA_KEYWORD_RETURN_ITALIC,
                PreferenceConstants.EDITOR_JAVA_KEYWORD_RETURN_UNDERLINE));

        returnKeywordTextStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_KEYWORD_RETURN).getRGB());
        return returnKeywordTextStyle;
    }

    @Override
    public TextStyle keywordTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.keywordTextStyle();
        }

        TextStyle javaKeywordStyle = defaultTextStyle();
        javaKeywordStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_JAVA_KEYWORD_BOLD,
                PreferenceConstants.EDITOR_JAVA_KEYWORD_ITALIC, PreferenceConstants.EDITOR_JAVA_KEYWORD_UNDERLINE));
        javaKeywordStyle.setColor(colorMan.getColor(IJavaColorConstants.JAVA_KEYWORD).getRGB());
        return javaKeywordStyle;
    }

    @Override
    public TextStyle taskTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.taskTextStyle();
        }

        TextStyle taskTextStyle = defaultTextStyle();
        taskTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_TASK_TAG_BOLD,
                PreferenceConstants.EDITOR_TASK_TAG_ITALIC, PreferenceConstants.EDITOR_TASK_TAG_UNDERLINE));
        taskTextStyle.setColor(colorMan.getColor(PreferenceConstants.EDITOR_TASK_TAG_COLOR).getRGB());
        return taskTextStyle;
    }

    @Override
    public TextStyle stringTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.stringTextStyle();
        }

        TextStyle stringTextStyle = defaultTextStyle();
        stringTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_STRING_BOLD,
                PreferenceConstants.EDITOR_STRING_ITALIC, PreferenceConstants.EDITOR_STRING_UNDERLINE));
        stringTextStyle.setColor(colorMan.getColor(PreferenceConstants.EDITOR_STRING_COLOR).getRGB());
        return stringTextStyle;
    }

    @Override
    public TextStyle numberTextStyle() {

        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.numberTextStyle();
        }
        TextStyle numberTextStyle = defaultTextStyle();
        numberTextStyle.setStyle(getJDTStyle(SemanticHighlightings.NUMBER));
        RGB color = getJDTStyleColor(SemanticHighlightings.NUMBER);
        numberTextStyle.setColor(color != null ? color : super.numberTextStyle().getColor());
        return numberTextStyle;
    }

    @Override
    public TextStyle defaultTextStyle() {
        if (!SemanticHighlightings.isEnabled(jdtPrefs)) {
            return super.defaultTextStyle();
        }
        TextStyle defaultTextStyle = super.defaultTextStyle();
        defaultTextStyle.setStyle(getJDTStyle(PreferenceConstants.EDITOR_JAVA_DEFAULT_BOLD,
                PreferenceConstants.EDITOR_JAVA_DEFAULT_ITALIC, PreferenceConstants.EDITOR_JAVA_DEFAULT_UNDERLINE));
        RGB color = colorMan.getColor(IJavaColorConstants.JAVA_DEFAULT).getRGB();
        defaultTextStyle.setColor(color);
        return defaultTextStyle;
    };

    private RGB getJDTStyleColor(String highlightKey) {
        SemanticHighlighting h = getSemHighlight(highlightKey);
        if (h == null) {
            return null;
        }
        RGB color = super.numberTextStyle().getColor();
        String enabledKey = SemanticHighlightings.getEnabledPreferenceKey(h);

        if (jdtPrefs.getBoolean(enabledKey)) {
            color = PreferenceConverter.getColor(jdtPrefs, SemanticHighlightings.getColorPreferenceKey(h));
        }
        return color;

    }

    private int getJDTStyle(String semHighlightId) {
        int style = SWT.NONE;

        SemanticHighlighting h = getSemHighlight(semHighlightId);
        if (h == null) {
            return style;
        }
        String enabledKey = SemanticHighlightings.getEnabledPreferenceKey(h);

        if (jdtPrefs.getBoolean(enabledKey)) {
            style = getJDTStyle(SemanticHighlightings.getBoldPreferenceKey(h),
                    SemanticHighlightings.getItalicPreferenceKey(h), SemanticHighlightings.getUnderlinePreferenceKey(h));
        }

        return style;
    }

    private int getJDTStyle(String bold, String italic, String underline) {
        int style = SWT.None;

        // set the BOLD, ITALIC and UNDERLINE style properties as they are in the jdt preference pages
        if (jdtPrefs.getBoolean(bold)) {
            style = style | SWT.BOLD;
        }
        if (jdtPrefs.getBoolean(italic)) {
            style = style | SWT.ITALIC;
        }
        if (jdtPrefs.getBoolean(underline)) {
            style = style | TextAttribute.UNDERLINE;
        }
        return style;
    }

    private SemanticHighlighting getSemHighlight(String key) {
        SemanticHighlighting[] highlights = SemanticHighlightings.getSemanticHighlightings();
        for (SemanticHighlighting h : highlights) {
            if (h.getPreferenceKey().equals(key)) {
                return h;
            }
        }
        return null;
    }

}
