/*******************************************************************************
 * Copyright (c) 2013 Stefan Prisca.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Stefan Prisca - initial API and implementation
 ******************************************************************************/
package org.eclipse.recommenders.templates.rcp.syntaxhighlighting;

import javax.inject.Inject;

import org.eclipse.recommenders.templates.services.TemplateGrammarAccess;
import org.eclipse.xtext.AbstractRule;
import org.eclipse.xtext.RuleCall;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.syntaxcoloring.IHighlightedPositionAcceptor;
import org.eclipse.xtext.ui.editor.syntaxcoloring.ISemanticHighlightingCalculator;

public class TemplatesSemanticHighlightingCalculator implements ISemanticHighlightingCalculator {

    private final TemplateGrammarAccess grammar;

    @Inject
    public TemplatesSemanticHighlightingCalculator(TemplateGrammarAccess grammar) {
        this.grammar = grammar;
    }

    @Override
    public void provideHighlightingFor(XtextResource resource, IHighlightedPositionAcceptor acceptor) {
        if (resource == null || resource.getParseResult() == null)
            return;

        int slCommentLine = -1, slCommentOffset = -1, mlCommentStartOffset = -1, mlCommentEndOffset;
        int stringStartOffset = -1, stringEndOffset = -1, stringStartLine = -1;
        String stringType = null;
        INode root = resource.getParseResult().getRootNode();
        mlCommentEndOffset = stringEndOffset = root.getTotalEndOffset() + 1;

        for (INode node : root.getAsTreeIterable()) {

            if (isDerived(node, grammar.getIDRule(), grammar.getTemplateIdentifierRule())
                    || isDerived(node, grammar.getJavaKeywordRule(), grammar.getTemplateIdentifierRule())
                    || isTemplateKeywordDerivedFrom(node, grammar.getTemplateIdentifierRule())
                    || isDerived(node, grammar.getSimpleVariableRule(), grammar.getTemplateIdentifierRule(),
                            grammar.getFullVariableRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.TEMPLATE_IDENTIFIER_ID);

            } else if (isDerived(node, grammar.getSimpleVariableRule())
                    && !isDerived(node, grammar.getSimpleVariableRule(), grammar.getTemplateIdentifierRule(),
                            grammar.getFullVariableRule())
                    && !isDerived(node, grammar.getSimpleVariableRule(), grammar.getTemplateIdentifierRule(),
                            grammar.getQualifiedNameRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.TEMPLATE_KEYWORD_ID);

            } else if (isDerived(node, grammar.getTypeRule(), grammar.getFullVariableRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.TEMPLATE_SPECIAL_TYPE_ID);

            } else if (isTemplateKeywordDerivedFrom(node, grammar.getFullVariableRule())
                    && !isTemplateKeywordDerivedFrom(node, grammar.getQualifiedNameRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.TEMPLATE_KEYWORD_ID);

            } else if (isDerived(node, grammar.getTemplateSingleQuotedStringRule(), grammar.getProposalRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(), TemplatesHighlightingConfiguration.STRING_ID);

            } else if (isDerived(node, grammar.getSingleQuotedStringRule())
                    && (stringType == null || node.getStartLine() != stringStartLine)
                    && !isStringInComment(node, slCommentOffset, slCommentLine, mlCommentStartOffset,
                            mlCommentEndOffset)) { // beginning of a SQ string

                stringStartOffset = node.getOffset();
                stringStartLine = node.getStartLine();
                stringEndOffset = root.getTotalEndOffset() + 1;
                stringType = grammar.getSingleQuotedStringRule().getName();

            } else if (isDerived(node, grammar.getSingleQuotedStringRule())
                    && stringType == grammar.getSingleQuotedStringRule().getName()) { // end of a SQ string

                stringEndOffset = node.getEndOffset();
                stringType = null;

            } else if (isDerived(node, grammar.getDoubleQuotedStringRule())
                    && (stringType == null || node.getStartLine() != stringStartLine)
                    && !isStringInComment(node, slCommentOffset, slCommentLine, mlCommentStartOffset,
                            mlCommentEndOffset)) { // beginning of a DQ string

                stringStartOffset = node.getOffset();
                stringStartLine = node.getStartLine();
                stringEndOffset = root.getTotalEndOffset() + 1;
                stringType = grammar.getDoubleQuotedStringRule().getName();

            } else if (isDerived(node, grammar.getDoubleQuotedStringRule())
                    && stringType == grammar.getDoubleQuotedStringRule().getName()) { // end of a DQ string

                stringEndOffset = node.getEndOffset();
                stringType = null;

            } else if (stringType != null && node.getStartLine() == stringStartLine
                    && node.getOffset() >= stringStartOffset && node.getOffset() - 1 <= stringEndOffset
                    && hasAncestor(node, grammar.getTextRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(), TemplatesHighlightingConfiguration.STRING_ID);

            } else if (isDerived(node, grammar.getSlCommentRule())
                    && !isCommentInString(stringStartOffset, node.getOffset(), node.getStartLine(), stringStartLine,
                            stringType != null)) {

                slCommentLine = node.getStartLine();
                slCommentOffset = node.getOffset();

            } else if (isDerived(node, grammar.getMlCommentStartRule())
                    && !isCommentInString(stringStartOffset, node.getOffset(), node.getStartLine(), stringStartLine,
                            stringType != null)

                    && !isMlCommentInSlComment(node.getStartLine(), slCommentLine, node.getOffset(), slCommentOffset)) {

                mlCommentStartOffset = node.getOffset();
                mlCommentEndOffset = root.getTotalEndOffset() + 1;

            } else if (isDerived(node, grammar.getMlCommentEndRule())) {

                mlCommentEndOffset = node.getEndOffset();
                mlCommentStartOffset = -1;

            } else if (node.getStartLine() == slCommentLine && hasAncestor(node, grammar.getTextRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(), TemplatesHighlightingConfiguration.JAVA_SINGLE_LINE_COMMENT_ID);

            } else if (mlCommentStartOffset >= 0 && node.getOffset() >= mlCommentStartOffset
                    && node.getEndOffset() <= mlCommentEndOffset && hasAncestor(node, grammar.getTextRule())) {

                acceptor.addPosition(node.getOffset(), node.getLength(), TemplatesHighlightingConfiguration.JAVA_MULTI_LINE_COMMENT_ID);

            } else if (isDerived(node, grammar.getJavaBracketsRule())) {
                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.JAVA_BRACKETS_ID);
            } else if (isDerived(node, grammar.getJavaOperatorRule())) {
                acceptor.addPosition(node.getOffset(), node.getLength(),
                        TemplatesHighlightingConfiguration.JAVA_OPERATOR_ID);
            }
        }
    }

    private boolean isMlCommentInSlComment(int startLine, int slCommentLine, int offset, int slCommentOffset) {
        return startLine == slCommentLine && offset > slCommentOffset;
    }

    private boolean isStringInComment(INode stringNode, int slCommentOffset, int slCommentLine,
            int mlCommentStartOffset, int mlCommentEndOffset) {
        int stringOffset = stringNode.getOffset();
        int stringLine = stringNode.getStartLine();
        return (slCommentOffset >= 0 && stringLine == slCommentLine && stringOffset > slCommentOffset)
                || (mlCommentStartOffset >= 0 && stringOffset > mlCommentStartOffset && stringOffset < mlCommentEndOffset);
    }

    private boolean isCommentInString(int stringStartOffset, int commentStartOffset, int commentStartLine,
            int stringStartLine, boolean openString) {
        return openString && commentStartOffset > stringStartOffset && commentStartLine == stringStartLine;
    }

    private boolean isTemplateKeywordDerivedFrom(INode node, AbstractRule ancestors) {
        return isDerived(node, grammar.getFieldTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getVarTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getLocalVarTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getArgTypeTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getElemTypeTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getNewNameTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getNewTypeTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getImportTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getImportStaticTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getLinkTemplateKeywordRule(), ancestors)
                || isDerived(node, grammar.getArrayTemplateKeywordRule(), ancestors);
    }

    /**
     * Checks whether the rule from node <b>node</b> is derived from the <b>ancestorRules</b> list. A rule has an
     * ancestor if it is reached, from the root element, through that ancestor. The list of ancestor rules respects the
     * following format: [nodeRule, parentRule, grandparentRule, grand-grandparentRule, etc]. <b>Note</b> that the first
     * element in the ancestor list must be the rule itself!
     *
     * @param node
     *            the node containing the rule for which to check ancestors
     * @param ancestorRules
     *            a list of expected ancestor rules given in the following format: [nodeRule, parentRule,
     *            grandparentRule, grand-grandparentRule, etc]
     * @return <b>true</b> if the rule derives from the expected ancestors. <b>false</b> otherwise
     */
    private boolean isDerived(INode node, AbstractRule... ancestorRules) {
        AbstractRule ancestorNodeRule;

        for (int i = 0; i < ancestorRules.length; i++) {

            if (node == null || !(node.getGrammarElement() instanceof RuleCall)) {
                return false;
            }

            ancestorNodeRule = ((RuleCall) node.getGrammarElement()).getRule();

            if (!(ancestorNodeRule.getName().equals(ancestorRules[i].getName()))) {
                return false;
            }

            node = node.getParent();
        }

        return true;
    }

    private boolean hasAncestor(INode node, AbstractRule ancestorRule) {

        if (node == null) {
            return false;
        }
        AbstractRule ancestorNodeRule;

        while (node != null) {

            if (node.getGrammarElement() instanceof RuleCall) {

                ancestorNodeRule = ((RuleCall) node.getGrammarElement()).getRule();

                if (ancestorNodeRule.getName().equals(ancestorRule.getName())) {
                    return true;
                }

            }

            node = node.getParent();

        }
        return false;
    }

}
