/**
 * Copyright (c) 2010, 2014 Darmstadt University of Technology.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *      Andreas Sewe - Initial design and API
 */
package org.eclipse.recommenders.templates.snipmatch.rcp;

import org.eclipse.core.databinding.beans.BeanProperties;
import org.eclipse.core.databinding.observable.ChangeEvent;
import org.eclipse.core.databinding.observable.IChangeListener;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.recommenders.snipmatch.ISnippet;
import org.eclipse.recommenders.snipmatch.Snippet;
import org.eclipse.recommenders.snipmatch.rcp.SnippetEditorInput;
import org.eclipse.recommenders.templates.rcp.internal.TemplateActivator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.xtext.resource.IResourceFactory;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.ui.editor.embedded.EmbeddedEditor;
import org.eclipse.xtext.ui.editor.embedded.EmbeddedEditorFactory;
import org.eclipse.xtext.ui.editor.embedded.EmbeddedEditorModelAccess;
import org.eclipse.xtext.ui.editor.embedded.IEditedResourceProvider;

import com.google.inject.Injector;

@SuppressWarnings("restriction")
public class SnippetSourcePage extends FormPage {

    private ISnippet snippet;
    private AbstractFormPart codePart;
    private EmbeddedEditor handle;

    public SnippetSourcePage(FormEditor editor, String id, String title) {
        super(editor, id, title);
    }

    @Override
    protected void createFormContent(IManagedForm managedForm) {
        FormToolkit toolkit = managedForm.getToolkit();
        ScrolledForm form = managedForm.getForm();
        form.setText(getTitle());
        Composite body = form.getBody();
        toolkit.decorateFormHeading(form.getForm());
        toolkit.paintBordersFor(body);
        managedForm.getForm().getBody().setLayout(new GridLayout());

        codePart = new CodePart();

        managedForm.addPart(codePart);

    }

    @Override
    public void init(IEditorSite site, IEditorInput input) {
        snippet = ((SnippetEditorInput) input).getSnippet();

        getEditor().addPropertyListener(new IPropertyListener() {

            @Override
            public void propertyChanged(Object source, int propId) {
                if (propId == PROP_INPUT) {
                    getManagedForm().removePart(codePart);
                    codePart.dispose();

                    snippet = ((SnippetEditorInput) getEditor().getEditorInput()).getSnippet();

                    codePart = new CodePart();
                    getManagedForm().addPart(codePart);

                }
            }
        });

        super.init(site, input);
    }

    class CodePart extends AbstractFormPart {

        IObservableValue snippetBeanCode;

        @Override
        public void initialize(IManagedForm form) {
            super.initialize(form);

            form.getForm().getBody()
                    .setLayoutData(GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER).grab(true, true));

            final TemplateActivator activator = TemplateActivator.getInstance();
            final Injector injector = activator
                    .getInjector(TemplateActivator.ORG_ECLIPSE_RECOMMENDERS_TEMPLATES_TEMPLATE);

            IEditedResourceProvider resourceProvider = new IEditedResourceProvider() {
                @Override
                public XtextResource createResource() {
                    IResourceFactory resourceFactory = injector.getInstance(IResourceFactory.class);
                    URI uri = URI.createURI("x-uuid:" + snippet.getUuid().toString());

                    XtextResourceSet rs = injector.getInstance(XtextResourceSet.class);
                    rs.setClasspathURIContext(getClass());

                    Resource resource = resourceFactory.createResource(uri);
                    rs.getResources().add(resource);
                    return (XtextResource) resource;
                }
            };
            EmbeddedEditorFactory factory = injector.getInstance(EmbeddedEditorFactory.class);
            handle = factory.newEditor(resourceProvider).showErrorAndWarningAnnotations()
                    .withParent(form.getForm().getBody());

            EmbeddedEditorModelAccess partialEditor = handle.createPartialEditor(true); // Needed

            handle.getDocument().set(snippet.getCode());
            handle.getDocument().addDocumentListener(new IDocumentListener() {

                @Override
                public void documentChanged(DocumentEvent event) {
                    String code = event.getDocument().get();
                    ((Snippet) snippet).setCode(code);
                }

                @Override
                public void documentAboutToBeChanged(DocumentEvent event) {
                    // No-op
                }
            });

            snippetBeanCode = BeanProperties.value(Snippet.class, "code", String.class).observe(snippet);

            snippetBeanCode.addChangeListener(new IChangeListener() {

                @Override
                public void handleChange(ChangeEvent event) {
                    if (!handle.getDocument().get().equals(snippet.getCode())) {
                        markStale();
                    } else {
                        markDirty();
                    }
                }
            });

        }

        @Override
        public void dispose() {
            snippetBeanCode.dispose();
            handle.getViewer().getControl().dispose();
            super.dispose();
        }

        public void updateInput() {

        }

        @Override
        public void commit(boolean onSave) {
            if (onSave) {
                super.commit(onSave);
            }
        }

        @Override
        public void refresh() {
            handle.getDocument().set(snippet.getCode());
            super.refresh();
        }
    };
}
