/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 */
package org.eclipse.recommenders.stats.rcp.persistence;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.utils.IOUtils;
import org.eclipse.recommenders.utils.gson.FieldNameTypeAdapter;
import org.eclipse.recommenders.utils.gson.FileTypeAdapter;
import org.eclipse.recommenders.utils.gson.ISO8601DateParser;
import org.eclipse.recommenders.utils.gson.MethodNameTypeAdapter;
import org.eclipse.recommenders.utils.gson.MultimapTypeAdapter;
import org.eclipse.recommenders.utils.gson.PackageNameTypeAdapter;
import org.eclipse.recommenders.utils.gson.TypeNameTypeAdapter;
import org.eclipse.recommenders.utils.gson.UuidTypeAdapter;
import org.eclipse.recommenders.utils.names.IFieldName;
import org.eclipse.recommenders.utils.names.IMethodName;
import org.eclipse.recommenders.utils.names.IPackageName;
import org.eclipse.recommenders.utils.names.ITypeName;
import org.eclipse.recommenders.utils.names.VmFieldName;
import org.eclipse.recommenders.utils.names.VmMethodName;
import org.eclipse.recommenders.utils.names.VmPackageName;
import org.eclipse.recommenders.utils.names.VmTypeName;
import org.osgi.framework.Bundle;
import org.osgi.framework.FrameworkUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Charsets;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.io.Files;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

public class DataPersistenceService implements IDataPersistenceService {

    private static final Logger LOG = LoggerFactory.getLogger(DataPersistenceService.class);

    @Override
    public <E> void record(E event, String channelId) {
        File dest = getCompletionLogLocation(channelId);
        if (event != null) {
            try {
                final Gson gson = initGsonObject(channelId);
                Files.append(gson.toJson(event) + IOUtils.LINE_SEPARATOR, dest, Charsets.UTF_8);
            } catch (IOException e) {
                LOG.debug(e.getMessage());
            }
        }
    }

    @Override
    public <E> Collection<E> retrieve(Class<E> eventClass, String channelId) {
        File dest = getCompletionLogLocation(channelId);
        List<E> events = Lists.newArrayList();
        if (dest.exists()) {
            Gson gson = initGsonObject(channelId);
            try {
                for (String json : Files.readLines(dest, Charsets.UTF_8)) {
                    E event = gson.fromJson(json, eventClass);
                    if (eventClass.isInstance(event)) {
                        events.add(event);
                    }
                }
            } catch (IOException e) {
                LOG.debug(e.getMessage());
            }
        }
        LOG.debug("Load data from file: " + dest.getPath());
        return events;
    }

    private File getCompletionLogLocation(String channelId) {
        Bundle bundle = FrameworkUtil.getBundle(DataPersistenceService.class);
        IPath location = Platform.getStateLocation(bundle);
        return new File(location.toFile(), channelId + ".txt");
    }

    private Gson initGsonObject(String channelId) {
        if (Constants.COMPLETION_EVENTS_CHANNEL.equals(channelId)) {
            final GsonBuilder builder = new GsonBuilder();
            builder.registerTypeAdapter(VmMethodName.class, new MethodNameTypeAdapter());
            builder.registerTypeAdapter(IMethodName.class, new MethodNameTypeAdapter());
            builder.registerTypeAdapter(VmTypeName.class, new TypeNameTypeAdapter());
            builder.registerTypeAdapter(ITypeName.class, new TypeNameTypeAdapter());
            builder.registerTypeAdapter(VmFieldName.class, new FieldNameTypeAdapter());
            builder.registerTypeAdapter(IFieldName.class, new FieldNameTypeAdapter());
            builder.registerTypeAdapter(VmPackageName.class, new PackageNameTypeAdapter());
            builder.registerTypeAdapter(IPackageName.class, new PackageNameTypeAdapter());
            builder.registerTypeAdapter(File.class, new FileTypeAdapter());
            builder.registerTypeAdapter(UUID.class, new UuidTypeAdapter());
            builder.registerTypeAdapter(Date.class, new ISO8601DateParser());
            builder.registerTypeAdapter(Multimap.class, new MultimapTypeAdapter());
            return builder.create();
        }
        return new Gson();
    }
}
