/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui;

import static java.lang.String.format;

import java.util.Collection;
import java.util.Date;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.ui.data.StatisticData;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.UiFormsFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.UiFormsSash;
import org.eclipse.recommenders.stats.rcp.ui.views.NavigationViewer;
import org.eclipse.recommenders.stats.rcp.ui.views.StatisticsView;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

import com.google.common.collect.Iterables;

public class DeveloperActivityDialog extends TitleAreaDialog {

    private final IDataPersistenceService persistenceService;

    public DeveloperActivityDialog(Shell parentShell, IDataPersistenceService persistenceService) {
        super(parentShell);
        this.persistenceService = persistenceService;
        setShellStyle(SWT.TITLE | SWT.MODELESS | SWT.RESIZE | SWT.CLOSE | SWT.BORDER);
        setHelpAvailable(false);
    }

    @Override
    protected Control createContents(Composite parent) {
        super.createContents(parent);
        getShell().setText("Developer Activity Report");
        getShell().setSize(750, 600);
        setTitle("Developer Activities");
        setMessage(getDescriptionText());
        return parent;
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        UiFormsFactory.createSeparator(parent);
        Composite mainComposite = TableViewerFactory.createWrapperComposite(parent);
        mainComposite.setBackground(new Color(Display.getCurrent(), 255, 255, 255));

        FormToolkit toolkit = new FormToolkit(mainComposite.getDisplay());
        UiFormsSash sashForm = UiFormsFactory.createSash(mainComposite, toolkit);

        Section masterSection = UiFormsFactory.createSection(sashForm, toolkit, "Navigation");
        Section detailSection = UiFormsFactory.createSection(sashForm, toolkit, "Eclipse Usage Statistics");
        final Composite masterCmp = UiFormsFactory.createCompositeForSection(toolkit, masterSection);
        final Composite detailCmp = UiFormsFactory.createCompositeForSection(toolkit, detailSection);
        sashForm.hookResizeListener();

        final StackLayout stack = new StackLayout();
        detailCmp.setLayout(stack);

        NavigationViewer navigationViewer = new NavigationViewer(masterCmp);
        navigationViewer.createViewer();

        navigationViewer.addSelectionChangedListener(stack, detailCmp, detailSection);
        stack.topControl = new StatisticsView(persistenceService).createContent(detailCmp);

        sashForm.setWeights(new int[] { 35, 65 });
        UiFormsFactory.createSeparator(parent);
        return parent;
    }

    @Override
    protected void createButtonsForButtonBar(Composite parent) {
        createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
    }

    @Override
    protected boolean isResizable() {
        return true;
    }

    private String getDescriptionText() {
        String date = "the beginning of recording";
        Collection<CompletionEvent> okayEvents = StatisticData.create(persistenceService).getOkayEvents();
        if (okayEvents.size() > 0) {
            Date start = new Date(Iterables.getFirst(okayEvents, null).sessionStarted);
            date = format("%tF", start);
        }
        return "Here is a summary of your activities since " + date;
    }
}
