/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.data;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.model.WorkbenchSession;
import org.eclipse.recommenders.stats.rcp.persistence.Constants;

import com.google.common.annotations.VisibleForTesting;
import com.ibm.icu.text.SimpleDateFormat;

public class WorkbenchData {

    private final IDataPersistenceService service;

    private WorkbenchData(IDataPersistenceService service) {
        this.service = service;
    }

    public static WorkbenchData create(IDataPersistenceService service) {
        return new WorkbenchData(service);
    }

    public Map<WeekDays, Long> getHoursForWeekdays() {
        Map<Long, Long> filteredRawData = filterSameDaysInRawData(getData());
        Map<WeekDays, List<Long>> rawData = getRawWeekDayData(filteredRawData);
        Map<WeekDays, Long> preparedData = getPreparedWeekDayData(rawData);

        return fillEmptyDays(preparedData);
    }

    public Map<WeekDays, List<Long>> getRawWeekDayData() {
        Map<Long, Long> filteredRawData = filterSameDaysInRawData(getData());
        return getRawWeekDayData(filteredRawData);
    }

    public Map<Long, Long> getRawData() {
        return filterSameDaysInRawData(getData());
    }

    @VisibleForTesting
    protected static Map<WeekDays, Long> fillEmptyDays(Map<WeekDays, Long> preparedData) {
        Map<WeekDays, Long> filledData = new TreeMap<WeekDays, Long>();
        for (WeekDays day : WeekDays.values()) {
            if (preparedData.containsKey(day)) {
                filledData.put(day, preparedData.get(day));
            } else {
                filledData.put(day, 0L);
            }
        }

        return filledData;
    }

    @VisibleForTesting
    protected static Map<WeekDays, Long> getPreparedWeekDayData(Map<WeekDays, List<Long>> rawWeekDayData) {
        Map<WeekDays, Long> preparedData = new HashMap<WeekDays, Long>();
        for (WeekDays day : rawWeekDayData.keySet()) {
            List<Long> timeList = rawWeekDayData.get(day);
            if (!timeList.isEmpty()) {
                long timeAvg = 0L;
                for (Long time : timeList) {
                    timeAvg += time;
                }
                long avg = timeAvg / timeList.size();
                preparedData.put(day, avg);
            }
        }
        return preparedData;
    }

    @VisibleForTesting
    protected static Map<Long, Long> filterSameDaysInRawData(Collection<WorkbenchSession> dataList) {
        // collect btw. filter sessions to associated days
        Map<String, List<WorkbenchSession>> daysToSessions = new HashMap<String, List<WorkbenchSession>>();
        if (dataList != null) {
            for (WorkbenchSession workbenchObject : dataList) {
                // day is through 'yyyyMMdd' unique
                String day = new SimpleDateFormat("yyyyMMdd", Locale.ENGLISH).format(workbenchObject.getStart());

                if (!daysToSessions.containsKey(day)) {
                    daysToSessions.put(day, new ArrayList<WorkbenchSession>());
                }
                // collect all sessions for this day
                daysToSessions.get(day).add(workbenchObject);
            }
        }

        // Map with start session (long) and the sum of the hours for this day (long)
        Map<Long, Long> dayToHourPerDay = new HashMap<Long, Long>();
        for (String day : daysToSessions.keySet()) {
            List<WorkbenchSession> list = daysToSessions.get(day);
            if (list.isEmpty()) {
                continue;
            }

            long hoursSum = 0L;
            for (WorkbenchSession session : list) {
                // calculate the sum (in hours) of all sessions for this day
                hoursSum += session.getEnd().getTime() - session.getStart().getTime();
            }
            dayToHourPerDay.put(list.get(0).getStart().getTime(), hoursSum);
        }

        return dayToHourPerDay;
    }

    @VisibleForTesting
    protected static Map<WeekDays, List<Long>> getRawWeekDayData(Map<Long, Long> dataList) {
        Map<WeekDays, List<Long>> weekDayMap = new HashMap<WeekDays, List<Long>>();
        if (dataList != null) {
            for (Long startTime : dataList.keySet()) {
                String day = new SimpleDateFormat("E", Locale.ENGLISH).format(startTime);

                WeekDays weekDay = WeekDays.getEnum(day);
                if (!weekDayMap.containsKey(weekDay)) {
                    weekDayMap.put(weekDay, new ArrayList<Long>());
                }

                long hoursPerDay = dataList.get(startTime);
                weekDayMap.get(weekDay).add(hoursPerDay);
            }
        }
        return weekDayMap;
    }

    private Collection<WorkbenchSession> getData() {
        return service.retrieve(WorkbenchSession.class, Constants.WORKBENCH_EVENTS_CHANNEL);
    }
}
