/**
 * Copyright (c) 2016 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Sash;
import org.eclipse.ui.forms.FormColors;
import org.eclipse.ui.forms.IFormColors;
import org.eclipse.ui.forms.IManagedForm;

public class UiFormsSash extends SashForm {

    private final List<Sash> sashes = new ArrayList<>();
    private final Listener listener = new Listener() {
        @Override
        public void handleEvent(final Event event) {
            switch (event.type) {
            case SWT.MouseEnter:
                event.widget.setData("hover", Boolean.TRUE); //$NON-NLS-1$
                ((Control) event.widget).redraw();
                break;
            case SWT.MouseExit:
                event.widget.setData("hover", null); //$NON-NLS-1$
                ((Control) event.widget).redraw();
                break;
            case SWT.Paint:
                onSashPaint(event);
                break;
            case SWT.Resize:
                hookSashListeners();
                break;
            default:
                break;
            }
        }
    };

    public UiFormsSash(final Composite parent, final int style) {
        super(parent, style);
    }

    @Override
    public void layout(final boolean changed) {
        super.layout(changed);
        hookSashListeners();
    }

    @Override
    public void layout(final Control[] children) {
        super.layout(children);
        hookSashListeners();
    }

    public void hookResizeListener() {
        Control[] children = getChildren();
        for (int i = 0; i < children.length; i++) {
            if (children[i] instanceof Sash) {
                continue;
            }
            children[i].addListener(SWT.Resize, listener);
        }
    }

    private void onSashPaint(final Event event) {
        final Sash sash = (Sash) event.widget;
        final IManagedForm form = (IManagedForm) sash.getParent().getData("form"); //$NON-NLS-1$
        final FormColors colors = form.getToolkit().getColors();
        final GC graphics = event.gc;
        final Boolean hover = (Boolean) sash.getData("hover"); //$NON-NLS-1$
        graphics.setBackground(colors.getColor(IFormColors.TB_BG));
        graphics.setForeground(colors.getColor(IFormColors.TB_BORDER));
        final Point size = sash.getSize();
        if (hover != null) {
            graphics.fillRectangle(0, 0, size.x, size.y);
        }
    }

    private void hookSashListeners() {
        purgeSashes();
        final Control[] children = getChildren();
        for (int i = 0; i < children.length; i++) {
            if (children[i] instanceof Sash) {
                final Sash sash = (Sash) children[i];
                if (sashes.contains(sash)) {
                    continue;
                }
                sash.addListener(SWT.Paint, listener);
                sash.addListener(SWT.MouseEnter, listener);
                sash.addListener(SWT.MouseExit, listener);
                sashes.add(sash);
            }
        }
    }

    private void purgeSashes() {
        for (final Iterator<?> iter = sashes.iterator(); iter.hasNext();) {
            final Sash sash = (Sash) iter.next();
            if (sash.isDisposed()) {
                iter.remove();
            }
        }
    }
}
