/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;
import static org.eclipse.recommenders.utils.Checks.cast;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.TreeSet;

import javax.inject.Inject;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent.ProposalKind;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.ui.data.StatisticData;
import org.eclipse.recommenders.stats.rcp.ui.util.DateFormatter;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.Completion;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.recommenders.utils.Bags;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;

import com.google.common.base.Predicate;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multiset.Entry;

public class CompletionTypeView implements IPageContent {

    private final Collection<CompletionEvent> okayEvents;
    private final IDataPersistenceService persistenceService;

    @Inject
    public CompletionTypeView(IDataPersistenceService persistenceService) {
        this.persistenceService = persistenceService;
        okayEvents = StatisticData.create(persistenceService).getOkayEvents();
    }

    @Override
    public Composite createContent(Composite detailParent) {
        Multiset<ProposalKind> proposalKindBag = HashMultiset.create();
        final Multimap<ProposalKind, CompletionEvent> multiMap = ArrayListMultimap.create();

        for (final ProposalKind kind : ProposalKind.values()) {
            Collection<CompletionEvent> byKind = Collections2.filter(okayEvents, new Predicate<CompletionEvent>() {
                @Override
                public boolean apply(CompletionEvent input) {
                    if (kind == input.applied) {
                        if (!multiMap.containsEntry(kind, input)) {
                            multiMap.put(kind, input);
                        }
                        return true;
                    }
                    return false;
                }
            });
            if (byKind.size() > 0) {
                proposalKindBag.add(kind, byKind.size());
            }
        }
        List<Entry<ProposalKind>> input = Bags.topUsingCount(proposalKindBag, 30);

        Composite newComp = createWrapperComposite(detailParent);
        LabelHelperFactory.createSeparatorLabel(newComp);

        LabelHelperFactory.createLabelWithColor(newComp, "Total number of triggered completion types: ",
                Integer.toString(getSize(input)));

        Composite comp = new Composite(newComp, SWT.NONE);
        TableColumnLayout layout = createTableColumnLayout(comp);

        final TableViewer viewer = createTableViewer(comp);
        TableViewerColumn completionTypeColumn = createColumn("Completion Type", viewer, 150, layout, 50, SWT.LEFT);
        TableViewerColumn usedCompletionColumn = createColumn("Used", viewer, 60, layout, 15, SWT.RIGHT);
        TableViewerColumn lastUsedColumn = createColumn("Last used", viewer, 110, layout, 35, SWT.RIGHT);

        TableSortConfigurator.newConfigurator(viewer, new Action() {
            @Override
            public void run() {
                viewer.refresh();
            };
        }).add(completionTypeColumn.getColumn(), Completion.TYPE_COMPARATOR)
                .add(usedCompletionColumn.getColumn(), Completion.USED_COMPARATOR)
                .add(lastUsedColumn.getColumn(), Completion.getLastUsedComparator(multiMap, persistenceService))
                .initialize(usedCompletionColumn.getColumn(), SWT.DOWN).configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new ProposalLabelProvider(multiMap));
        viewer.setInput(input);

        LabelHelperFactory.createCounter(newComp, input.size());

        return newComp;
    }

    private int getSize(List<Entry<ProposalKind>> input) {
        int result = 0;
        for (Entry<ProposalKind> entry : input) {
            result += entry.getCount();
        }
        return result;
    }

    public class ProposalLabelProvider extends StyledCellLabelProvider {
        private Multimap<ProposalKind, CompletionEvent> eventsByKind;

        public ProposalLabelProvider(Multimap<ProposalKind, CompletionEvent> multiMap) {
            eventsByKind = multiMap;
        }

        @Override
        public void update(ViewerCell cell) {
            final StyledString cellText = new StyledString();
            Entry<ProposalKind> element = cast(cell.getElement());
            ProposalKind kind = element.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText.append(kind.toString().toLowerCase().replace('_', ' '));
                break;
            case 1:
                int count = element.getCount();
                cellText.append(Integer.toString(count), StyledString.COUNTER_STYLER);
                break;
            case 2:
                Date past = new Date(getLastSessionStartedFor(kind));
                cellText.append(new DateFormatter().formatUnit(past, new Date()));
                break;
            }

            cell.setText(cellText.toString());
            cell.setStyleRanges(cellText.getStyleRanges());
        }

        public Long getLastSessionStartedFor(ProposalKind proposal) {
            Collection<CompletionEvent> collection = eventsByKind.get(proposal);
            TreeSet<Long> sessionSet = new TreeSet<Long>();
            for (CompletionEvent completionEvent : collection) {
                sessionSet.add(completionEvent.sessionEnded);
            }
            return sessionSet.last();
        }
    }
}
