/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.TriggeredCommandObject;
import org.eclipse.recommenders.stats.rcp.ui.data.CommandData;
import org.eclipse.recommenders.stats.rcp.ui.data.WorkbenchData;
import org.eclipse.recommenders.stats.rcp.ui.util.CommandsUtil;
import org.eclipse.recommenders.stats.rcp.ui.util.EclipseCategoryData;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

public class EclipseCategoryView implements IPageContent {

    private final IDataPersistenceService persistenceService;
    private final Collection<TriggeredCommandObject> commandData;
    private final Map<Long, Long> workbenchData;
    private Composite parent;

    @Inject
    public EclipseCategoryView(IDataPersistenceService persistenceService) {
        this.persistenceService = persistenceService;
        commandData = CommandData.create(persistenceService).getData();
        workbenchData = WorkbenchData.create(persistenceService).getRawData();
    }

    @Override
    public Composite createContent(Composite detailParent) {
        createComposite(detailParent);
        createOverviewSection();
        createCommandSection();
        createActivitySection();
        return parent;
    }

    private void createComposite(Composite detailParent) {
        parent = TableViewerFactory.createWrapperComposite(detailParent);
        GridLayout grid = new GridLayout(2, false);
        grid.marginHeight = 5;
        grid.horizontalSpacing = 0;
        grid.verticalSpacing = 0;
        parent.setLayout(grid);
    }

    private void createOverviewSection() {
        LabelHelperFactory.createSeparatorLabel(parent);
        createFilledLabel(parent, "Overview about commands usage");
        new Label(parent, SWT.NONE).setText("Number of times commands triggered: ");
        LabelHelperFactory.createLabelWithColor(parent, CommandsUtil.getTriggeredCommandsSize(commandData) + "");
        new Label(parent, SWT.NONE).setText("Beginning of activity tracing: ");
        LabelHelperFactory.createLabelWithColor(parent, CommandsUtil.getDescriptionText(persistenceService));
    }

    private void createCommandSection() {
        LabelHelperFactory.createSeparatorLabel(parent);
        List<TriggeredCommandObject> commandList = EclipseCategoryData.getFavouriteCommands(commandData);
        if (commandList.isEmpty()) {
            createFilledLabel(parent, "Currently no commands tracked.");
        } else {
            createFilledLabel(parent, "Your favourite commands in this IDE");
            new Label(parent, SWT.NONE).setText("Command #1: ");
            LabelHelperFactory.createLabelWithColor(parent, CommandsUtil.getCommandName(commandList.get(0)));
            new Label(parent, SWT.NONE).setText("Command #2: ");
            LabelHelperFactory.createLabelWithColor(parent, CommandsUtil.getCommandName(commandList.get(1)));
            new Label(parent, SWT.NONE).setText("Command #3: ");
            LabelHelperFactory.createLabelWithColor(parent, CommandsUtil.getCommandName(commandList.get(2)));
        }
    }

    private void createActivitySection() {
        LabelHelperFactory.createSeparatorLabel(parent);
        createFilledLabel(parent, "Number of hours spent in Eclipse");
        new Label(parent, SWT.NONE).setText("Total number of hours: ");
        LabelHelperFactory.createLabelWithColor(parent,
                EclipseCategoryData.getTotalNumberHours(workbenchData) + " hours");
        new Label(parent, SWT.NONE).setText("Min. number of hours: ");
        LabelHelperFactory.createLabelWithColor(parent,
                EclipseCategoryData.getMinHoursPerDay(workbenchData) + " hours");
        new Label(parent, SWT.NONE).setText("Max. number of hours: ");
        LabelHelperFactory.createLabelWithColor(parent,
                EclipseCategoryData.getMaxHoursPerDay(workbenchData) + " hours");
    }

    private void createFilledLabel(Composite container, String text) {
        Label label = new Label(container, SWT.NONE);
        label.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
        label.setText(text);
    }
}
