/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;
import static org.eclipse.recommenders.utils.Checks.cast;

import java.util.Collection;
import java.util.List;

import javax.inject.Inject;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.ui.data.StatisticData;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.ReceiverType;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.recommenders.utils.Bags;
import org.eclipse.recommenders.utils.names.ITypeName;
import org.eclipse.recommenders.utils.names.Names;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multiset.Entry;

public class ReceiverTypeView implements IPageContent {

    private final Collection<CompletionEvent> okayEvents;

    @Inject
    public ReceiverTypeView(IDataPersistenceService persistenceService) {
        okayEvents = StatisticData.create(persistenceService).getOkayEvents();
    }

    @Override
    public Composite createContent(Composite detailParent) {
        Multiset<ITypeName> b = HashMultiset.create();
        for (CompletionEvent e : okayEvents) {
            if (e.receiverType == null) {
                continue;
            }
            b.add(e.receiverType);
        }
        List<Entry<ITypeName>> input = Bags.topUsingCount(b, 30);

        Composite newComp = createWrapperComposite(detailParent);
        LabelHelperFactory.createSeparatorLabel(newComp);

        LabelHelperFactory.createLabelWithColor(newComp, "Total number of triggered receiver types: ",
                Integer.toString(getSize(input)));

        Composite comp = new Composite(newComp, SWT.NONE);
        TableColumnLayout layout = createTableColumnLayout(comp);

        final TableViewer viewer = createTableViewer(comp);
        TableViewerColumn typeColumn = createColumn("Type", viewer, 450, layout, 60, SWT.LEFT);
        TableViewerColumn countColumn = createColumn("Count", viewer, 100, layout, 30, SWT.RIGHT);

        TableSortConfigurator.newConfigurator(viewer, new Action() {
            @Override
            public void run() {
                viewer.refresh();
            };
        }).add(typeColumn.getColumn(), ReceiverType.TYPE_COMPARATOR)
                .add(countColumn.getColumn(), ReceiverType.COUNT_COMPARATOR)
                .initialize(countColumn.getColumn(), SWT.DOWN).configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new TypeNameLabelProvider());
        viewer.setInput(input);

        LabelHelperFactory.createCounter(newComp, input.size());

        return newComp;
    }

    private int getSize(List<Entry<ITypeName>> input) {
        int result = 0;
        for (Entry<ITypeName> entry : input) {
            result += entry.getCount();
        }
        return result;
    }

    public class TypeNameLabelProvider extends StyledCellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            final StyledString cellText = new StyledString();
            Entry<ITypeName> e = cast(cell.getElement());

            switch (cell.getColumnIndex()) {
            case 0:
                cellText.append(Names.vm2srcQualifiedType(e.getElement()));
                break;
            case 1:
                cellText.append(Integer.toString(e.getCount()), StyledString.COUNTER_STYLER);
                break;
            }

            cell.setText(cellText.toString());
            cell.setStyleRanges(cellText.getStyleRanges());
        }
    }
}
