/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static java.lang.Math.round;
import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.*;
import static org.apache.commons.math.stat.StatUtils.*;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.ui.data.StatisticData;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

public class StatisticsView implements IPageContent {

    private static long MAX_TIME_IN_COMPLETION = TimeUnit.MINUTES.toMillis(2);
    private final Collection<CompletionEvent> okayEvents;
    private final Collection<CompletionEvent> appliedEvents;
    private final Collection<CompletionEvent> abortedEvents;
    private Composite container;
    private Composite parent;

    @Inject
    public StatisticsView(IDataPersistenceService persistenceService) {
        StatisticData statsData = StatisticData.create(persistenceService);
        okayEvents = statsData.getOkayEvents();
        appliedEvents = statsData.getAppliedEvents();
        abortedEvents = statsData.getAbortedEvents();
    }

    @Override
    public Composite createContent(Composite detailCmp) {
        parent = TableViewerFactory.createWrapperComposite(detailCmp);
        createWidgets(parent);
        createNumberOfCompletionEvents();
        createNumberOfKeystrokesSaved();
        createTimeSpent();
        return parent;
    }

    private void createWidgets(Composite parent) {
        container = new Composite(parent, SWT.NONE);
        GridLayout grid = new GridLayout(2, false);
        grid.marginHeight = 5;
        grid.horizontalSpacing = 0;
        grid.verticalSpacing = 0;
        container.setLayout(grid);
    }

    private void createNumberOfCompletionEvents() {
        int total = 0;
        for (CompletionEvent e : okayEvents) {
            total += e.numberOfProposals;
        }

        LabelHelperFactory.createTwoSeparatorLabel(container);
        int completedInPercent = calculatePercentData(appliedEvents);
        new Label(container, SWT.NONE).setText("Number of times code completion triggered: ");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", okayEvents.size()));
        int abortedInPercent = calculatePercentData(abortedEvents);

        new Label(container, SWT.NONE).setText("Number of concluded completions: ");
        LabelHelperFactory.createLabelWithColor(container,
                MessageFormat.format("{0} ({1}%)", appliedEvents.size(), completedInPercent));

        new Label(container, SWT.NONE).setText("Number of aborted completions: ");
        LabelHelperFactory.createLabelWithColor(container,
                MessageFormat.format("{0} ({1}%)", abortedEvents.size(), abortedInPercent));

        new Label(container, SWT.NONE).setText("Number of proposals offered by code completion: ");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", total));
    }

    private void createNumberOfKeystrokesSaved() {
        double[] strokes = new double[appliedEvents.size()];
        int i = 0;
        for (CompletionEvent e : appliedEvents) {
            int prefix = e.prefix == null ? 0 : e.prefix.length();
            int completionLength = e.completion == null ? 0 : e.completion.length();
            int saved = Math.max(0, completionLength - prefix);
            strokes[i++] = saved;
        }

        Double total = sum(strokes);

        new Label(container, SWT.NONE).setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
        createFilledLabel("Keystrokes saved by using code completion");

        new Label(container, SWT.NONE).setText("Total number: ");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", total.isNaN() ? 0 : total));

        Double mean = mean(strokes);
        new Label(container, SWT.NONE).setText("Average number:");
        LabelHelperFactory.createLabelWithColor(container,
                MessageFormat.format("{0,number,#.##}", mean.isNaN() ? 0 : mean));
    }

    private void createTimeSpent() {
        double[] spentApplied = computeTimeSpentInCompletion(appliedEvents);
        long totalApplied = round(sum(spentApplied));
        long meanApplied = round(mean(spentApplied));

        double[] spentAborted = computeTimeSpentInCompletion(abortedEvents);
        long totalAborted = round(sum(spentAborted));
        long meanAborted = round(mean(spentAborted));

        new Label(container, SWT.NONE).setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
        createFilledLabel("Total Time spent in completion window on");

        new Label(container, SWT.NONE).setText("Concluded sessions:");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", toTimeString(totalApplied)));

        new Label(container, SWT.NONE).setText("Aborted sessions:");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", toTimeString(totalAborted)));

        new Label(container, SWT.NONE).setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
        createFilledLabel("Average time spent in completion window per");

        new Label(container, SWT.NONE).setText("Concluded sessions:");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", meanApplied));

        new Label(container, SWT.NONE).setText("Aborted sessions:");
        LabelHelperFactory.createLabelWithColor(container, MessageFormat.format("{0}", meanAborted));
    }

    private int calculatePercentData(Collection<CompletionEvent> list) {
        if (okayEvents.size() == 0) {
            return okayEvents.size();
        }
        double division = list.size() / (double) okayEvents.size() * 100;
        return (int) Math.round(division);
    }

    private void createFilledLabel(String text) {
        Label label2 = new Label(container, SWT.NONE);
        label2.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 2, 1));
        label2.setText(text);
    }

    private String toTimeString(long time) {
        return format("%d min, %d sec", MILLISECONDS.toMinutes(time),
                MILLISECONDS.toSeconds(time) - MINUTES.toSeconds(MILLISECONDS.toMinutes(time)));
    }

    private double[] computeTimeSpentInCompletion(Collection<CompletionEvent> events) {
        double[] spent = new double[events.size()];
        int i = 0;
        for (CompletionEvent e : events) {
            long ms = e.sessionEnded - e.sessionStarted;
            if (ms > MAX_TIME_IN_COMPLETION) {
                ms = MAX_TIME_IN_COMPLETION;
            }
            spent[i++] = ms;
        }
        return spent;
    }
}
