/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;

import java.util.Collection;

import javax.inject.Inject;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.TriggeredCommandObject;
import org.eclipse.recommenders.stats.rcp.ui.data.CommandData;
import org.eclipse.recommenders.stats.rcp.ui.util.CommandsUtil;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.Commands;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;

public class TriggeredCommandsView implements IPageContent {

    private Composite parent;
    private final IDataPersistenceService persistenceService;

    @Inject
    public TriggeredCommandsView(IDataPersistenceService persistenceService) {
        this.persistenceService = persistenceService;
    }

    @Override
    public Composite createContent(Composite detailCmp) {
        parent = TableViewerFactory.createWrapperComposite(detailCmp);

        LabelHelperFactory.createSeparatorLabel(parent);
        final Collection<TriggeredCommandObject> model = CommandData.create(persistenceService).getData();
        LabelHelperFactory.createLabelWithColor(parent, "Number of times commands triggered: ",
                Integer.toString(CommandsUtil.getTriggeredCommandsSize(model)));

        final Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));

        final TableColumnLayout layout = createTableColumnLayout(comp);
        final TableViewer viewer = createTableViewer(comp);

        final TableViewerColumn commandNameColumn = createColumn("Command Name", viewer, 120, layout, 20, SWT.LEFT);
        TableViewerColumn countColumn = createColumn("Count", viewer, 80, layout, 8, SWT.RIGHT);
        TableViewerColumn commandIdColumn = createColumn("Command ID", viewer, 130, layout, 40, SWT.LEFT);
        commandIdColumn.getColumn().setWidth(0);
        createColumn("Description", viewer, 130, layout, 40, SWT.LEFT);

        Action refreshUiAction = new Action() {
            @Override
            public void run() {
                viewer.refresh();
            }
        };

        CommandsLabelProvider labelProvider = new CommandsLabelProvider();
        TableSortConfigurator.newConfigurator(viewer, refreshUiAction)
                .add(commandNameColumn.getColumn(), Commands.getNameComparator(labelProvider))
                .add(countColumn.getColumn(), Commands.COUNT_COMPARATOR)
                .add(commandIdColumn.getColumn(), Commands.ID_COMPARATOR).initialize(countColumn.getColumn(), SWT.DOWN)
                .configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(labelProvider);
        viewer.setInput(model);

        LabelHelperFactory.createCounter(parent, model.size());

        return parent;
    }

    public class CommandsLabelProvider extends StyledCellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            if (isNotValidCommandObject(cell)) {
                return;
            }

            final StyledString cellText = new StyledString();
            TriggeredCommandObject command = (TriggeredCommandObject) cell.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText.append(CommandsUtil.getCommandName(command));
                break;
            case 1:
                cellText.append(Integer.toString(command.count), StyledString.COUNTER_STYLER);
                break;
            case 2:
                cellText.append(command.commandID);
                break;
            case 3:
                cellText.append(CommandsUtil.getCommandDescription(command));
                break;
            default:
            }

            cell.setText(cellText.toString());
            cell.setStyleRanges(cellText.getStyleRanges());
        }

        private boolean isNotValidCommandObject(ViewerCell cell) {
            return !(cell.getElement() instanceof TriggeredCommandObject);
        }
    }
}
