/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;

import java.util.List;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceProjectType;
import org.eclipse.recommenders.stats.rcp.ui.data.WorkspaceProjectData;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.WorkspaceProjects;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

public class WorkspaceProjectView implements IPageContent {

    private Label numberOfProjectsDesc;
    private Label sizeOfProjectsDesc;
    private Label projectNumberLabel;
    private Label projectSizeLabel;
    private TableViewer viewer;
    private Label counterLabel;

    @Override
    public Composite createContent(Composite detailParent) {
        final Composite parent = TableViewerFactory.createWrapperComposite(detailParent);

        Composite labelCmp = createWrapper(parent);
        LabelHelperFactory.createTwoSeparatorLabel(labelCmp);
        numberOfProjectsDesc = new Label(labelCmp, SWT.NONE);
        numberOfProjectsDesc.setText("Loading Data from Workspace, please wait ...");
        projectNumberLabel = LabelHelperFactory.createLabelWithColor(labelCmp);
        sizeOfProjectsDesc = new Label(labelCmp, SWT.NONE);
        sizeOfProjectsDesc.setText("");
        projectSizeLabel = LabelHelperFactory.createLabelWithColor(labelCmp);

        final Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));
        final TableColumnLayout layout = createTableColumnLayout(comp);
        viewer = createVirtualTableViewer(comp);

        TableViewerColumn nameCol = createColumn("Name", viewer, 80, layout, 30, SWT.CENTER);
        TableViewerColumn typeCol = createColumn("Type", viewer, 60, layout, 15, SWT.CENTER);
        TableViewerColumn filesCol = createColumn("Files", viewer, 30, layout, 10, SWT.CENTER);
        TableViewerColumn sizeCol = createColumn("Size", viewer, 30, layout, 10, SWT.CENTER);

        TableSortConfigurator.newConfigurator(viewer, new Action() {
            @Override
            public void run() {
                viewer.refresh();
            };
        }).add(nameCol.getColumn(), WorkspaceProjects.NAME_COMPARATOR)
                .add(typeCol.getColumn(), WorkspaceProjects.TYPE_COMPARATOR)
                .add(filesCol.getColumn(), WorkspaceProjects.COUNT_COMPARATOR)
                .add(sizeCol.getColumn(), WorkspaceProjects.SIZE_COMPARATOR).initialize(sizeCol.getColumn(), SWT.DOWN)
                .configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setLabelProvider(new WorkspaceProjectLabelProvider());
        viewer.setContentProvider(new ArrayContentProvider());

        counterLabel = LabelHelperFactory.createCounter(parent, 0);
        startProjectCollectorJob();

        return parent;
    }

    private void startProjectCollectorJob() {
        final Job findFilesJob = new Job("Collect all Projects") {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask("Collect Projects from Workspace", IProgressMonitor.UNKNOWN);

                WorkspaceProjectData workspaceData = new WorkspaceProjectData();
                List<WorkspaceProjectType> allProjects = workspaceData.readAllProjects();
                updateUI(workspaceData, allProjects);

                if (monitor.isCanceled()) {
                    return Status.CANCEL_STATUS;
                }
                return Status.OK_STATUS;
            }

            private void updateUI(final WorkspaceProjectData workspaceData,
                    final List<WorkspaceProjectType> allProjects) {
                Display.getDefault().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        numberOfProjectsDesc.setText("Total Number of all Projects: ");
                        sizeOfProjectsDesc.setText("File Size of these Projects: ");

                        String fileSize = FileUtils.byteCountToDisplaySize(workspaceData.getTotalSize());
                        projectNumberLabel.setText(Integer.toString(workspaceData.getTotalProjects()));
                        projectSizeLabel.setText(fileSize);
                        projectNumberLabel.getParent().pack();

                        viewer.setInput(allProjects);
                        LabelHelperFactory.update(counterLabel, allProjects.size());
                    }
                });
            }
        };
        findFilesJob.setPriority(Job.SHORT);
        findFilesJob.schedule();
    }

    private Composite createWrapper(final Composite parent) {
        Composite labelCmp = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        layout.marginWidth = 0;
        layout.marginHeight = 0;
        labelCmp.setLayout(layout);
        labelCmp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        return labelCmp;
    }

    private class WorkspaceProjectLabelProvider extends StyledCellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            final StyledString cellText = new StyledString();
            WorkspaceProjectType workspaceProject = (WorkspaceProjectType) cell.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText.append(workspaceProject.getName());
                break;
            case 1:
                cellText.append(workspaceProject.getType());
                break;
            case 2:
                cellText.append(workspaceProject.getFileCount(), StyledString.COUNTER_STYLER);
                break;
            case 3:
                cellText.append(FileUtils.byteCountToDisplaySize(workspaceProject.getBytes()));
                break;
            }

            cell.setText(cellText.toString());
            cell.setStyleRanges(cellText.getStyleRanges());
        }
    }
}
