/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;

import java.util.List;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceContentType;
import org.eclipse.recommenders.stats.rcp.ui.data.WorkspaceFileData;
import org.eclipse.recommenders.stats.rcp.ui.util.LabelHelperFactory;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.WorkspaceFiles;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

public class WorkspaceView implements IPageContent {

    private Label numberOfFilesDesc;
    private Label numberOfFileSizeDesc;
    private Label fileSizeCounter;
    private Label fileNumberCounter;
    private TableViewer viewer;
    private Label counterLabel;

    @Override
    public Composite createContent(Composite detailParent) {
        final Composite parent = TableViewerFactory.createWrapperComposite(detailParent);

        Composite labelCmp = createWrapper(parent);
        LabelHelperFactory.createTwoSeparatorLabel(labelCmp);
        numberOfFilesDesc = new Label(labelCmp, SWT.NONE);
        numberOfFilesDesc.setText("Loading Data from Workspace, please wait ...");
        fileNumberCounter = LabelHelperFactory.createLabelWithColor(labelCmp);
        numberOfFileSizeDesc = new Label(labelCmp, SWT.NONE);
        numberOfFileSizeDesc.setText("");
        fileSizeCounter = LabelHelperFactory.createLabelWithColor(labelCmp);

        final Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));

        final TableColumnLayout layout = createTableColumnLayout(comp);
        viewer = createVirtualTableViewer(comp);

        TableViewerColumn typeColumn = createColumn("File Type", viewer, 60, layout, 10, SWT.CENTER);
        TableViewerColumn countColumn = createColumn("Count", viewer, 60, layout, 10, SWT.CENTER);
        TableViewerColumn sizeColumn = createColumn("Size", viewer, 60, layout, 10, SWT.CENTER);

        TableSortConfigurator.newConfigurator(viewer, new Action() {
            @Override
            public void run() {
                viewer.refresh();
            };
        }).add(typeColumn.getColumn(), WorkspaceFiles.TYPE_COMPARATOR)
                .add(countColumn.getColumn(), WorkspaceFiles.COUNT_COMPARATOR)
                .add(sizeColumn.getColumn(), WorkspaceFiles.SIZE_COMPARATOR)
                .initialize(sizeColumn.getColumn(), SWT.DOWN).configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setLabelProvider(new WorkspaceFileLabelProvider());
        viewer.setContentProvider(new ArrayContentProvider());

        counterLabel = LabelHelperFactory.createCounter(parent, 0);
        startFileCollectorJob();

        return parent;
    }

    private void startFileCollectorJob() {
        final Job findFilesJob = new Job("Collect all Files") {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask("Collect Files from Workspace", IProgressMonitor.UNKNOWN);

                WorkspaceFileData workspaceData = new WorkspaceFileData();
                List<WorkspaceContentType> preparedData = workspaceData.getPreparedData();
                updateUI(workspaceData, preparedData);

                if (monitor.isCanceled()) {
                    return Status.CANCEL_STATUS;
                }
                return Status.OK_STATUS;
            }

            private void updateUI(final WorkspaceFileData workspaceData,
                    final List<WorkspaceContentType> preparedData) {
                Display.getDefault().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        numberOfFilesDesc.setText("Total Number of all Files: ");
                        numberOfFileSizeDesc.setText("File Size of these Files: ");

                        String fileSize = FileUtils.byteCountToDisplaySize(workspaceData.getTotalSize());
                        fileNumberCounter.setText(Integer.toString(workspaceData.getTotalFiles()));
                        fileSizeCounter.setText(fileSize);

                        numberOfFilesDesc.getParent().pack();
                        viewer.setInput(preparedData);

                        LabelHelperFactory.update(counterLabel, preparedData.size());
                    }
                });
            }
        };
        findFilesJob.setPriority(Job.SHORT);
        findFilesJob.schedule();
    }

    private Composite createWrapper(final Composite parent) {
        Composite labelCmp = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        layout.marginWidth = 0;
        layout.marginHeight = 0;
        labelCmp.setLayout(layout);
        labelCmp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        return labelCmp;
    }

    private class WorkspaceFileLabelProvider extends StyledCellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            final StyledString cellText = new StyledString();
            WorkspaceContentType workspaceFile = (WorkspaceContentType) cell.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText.append(workspaceFile.getLabel());
                break;
            case 1:
                cellText.append(workspaceFile.getCount(), StyledString.COUNTER_STYLER);
                break;
            case 2:
                cellText.append(FileUtils.byteCountToDisplaySize(workspaceFile.getBytes()));
                break;
            }

            cell.setText(cellText.toString());
            cell.setStyleRanges(cellText.getStyleRanges());
        }
    }
}
