/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.charts.rcp.charts;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.birt.chart.model.Chart;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.recommenders.stats.charts.rcp.renderer.AbstractChartPage;
import org.eclipse.recommenders.stats.rcp.ui.data.GitCommitData;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

import com.google.common.collect.Iterables;

public class GitCommitChart extends AbstractChartPage {

    private static final String COMMIT_LABEL = "Total number of commits: ";

    @Override
    public Chart createChart() {

        Set<String> gitRepositories = GitCommitData.readAllGitRepositories();
        if (gitRepositories.isEmpty()) {
            return BarChartFactory.create("No Git repositories found in Eclipse workspace", "", new String[] {},
                    new Double[] {});
        }

        final String firstRepo = Iterables.getFirst(gitRepositories, "");
        final Map<String, Double> commitMap = new HashMap<>();
        readCommitsInBackground(firstRepo, commitMap);

        new Label(getParent(), SWT.NONE).setText("Please select here one of your configured GIT repositories.");
        final ComboViewer combo = new ComboViewer(getParent(), SWT.READ_ONLY);
        final Label commitLabel = new Label(getParent(), SWT.NONE);
        commitLabel.setText(COMMIT_LABEL + GitCommitData.getCommitNumber(commitMap));

        combo.setContentProvider(new ArrayContentProvider());
        combo.setLabelProvider(new ComboLabelProvider());

        combo.setInput(gitRepositories);
        combo.setSelection(new StructuredSelection(firstRepo));
        combo.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
                ISelection selection = combo.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection comboSelection = (IStructuredSelection) selection;
                    String firstElement = (String) comboSelection.getFirstElement();

                    readCommitsInBackground(firstElement, commitMap);
                    Chart chart = createBarChart(GitCommitData.sortByValue(commitMap));
                    commitLabel.setText(COMMIT_LABEL + GitCommitData.getCommitNumber(commitMap));

                    setChart(chart);
                    getParent().redraw();
                }
            }
        });

        return createBarChart(GitCommitData.sortByValue(commitMap));
    }

    private void readCommitsInBackground(final String firstRepo, final Map<String, Double> commitMap) {
        BusyIndicator.showWhile(Display.getCurrent(), new Runnable() {
            @Override
            public void run() {
                commitMap.clear();
                commitMap.putAll(GitCommitData.readAllCommits(firstRepo));
            }
        });
    }

    private Chart createBarChart(Map<String, Double> commitMap) {
        String[] names = commitMap.keySet().toArray(new String[commitMap.size()]);
        Double[] values = commitMap.values().toArray(new Double[commitMap.size()]);
        return BarChartFactory.create("Git Commit Overview", "# Commits", names, values);
    }

    private class ComboLabelProvider extends LabelProvider {
        @Override
        public String getText(Object element) {
            if (element instanceof String) {
                return GitCommitData.extractRepoName((String) element);
            }
            return super.getText(element);
        }
    }
}
