/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.data;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FilenameUtils;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceContentType;

import com.google.common.annotations.VisibleForTesting;

public class WorkspaceFileData {

    private int totalFileSize;
    private long totalByteSize;

    public List<WorkspaceContentType> getPreparedData() {
        List<File> collectedFiles = readFiles();
        Map<String, List<File>> dataMap = prepareData(collectedFiles);
        totalFileSize = collectedFiles.size();
        totalByteSize = calculateTotalSize(dataMap);

        return buildModelObjects(dataMap);
    }

    private List<File> readFiles() {
        List<File> collectedFiles = new ArrayList<File>();
        IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
        for (IProject project : workspaceRoot.getProjects()) {
            if (project.exists()) {
                IPath path = project.getLocation();
                WorkspaceDataHelper.collectAllFilesFor(path.toFile(), collectedFiles);
            }
        }
        return collectedFiles;
    }

    @VisibleForTesting
    protected Map<String, List<File>> prepareData(List<File> collectedFiles) {
        Map<String, List<File>> dataMap = new HashMap<String, List<File>>();

        if (collectedFiles != null) {
            for (File file : collectedFiles) {
                String extension = FilenameUtils.getExtension(file.getName());
                if (extension != null && !extension.isEmpty()) {
                    if (!dataMap.containsKey(extension)) {
                        dataMap.put(extension, new ArrayList<File>());
                    }
                    dataMap.get(extension).add(file);
                }
            }
        }
        return dataMap;
    }

    @VisibleForTesting
    protected List<WorkspaceContentType> buildModelObjects(Map<String, List<File>> dataMap) {
        List<WorkspaceContentType> modelList = new ArrayList<WorkspaceContentType>();

        if (dataMap != null) {
            for (String extension : dataMap.keySet()) {
                List<File> list = dataMap.get(extension);
                modelList.add(new WorkspaceContentType(extension, list.size(), WorkspaceDataHelper.getSizeFor(list)));
            }
        }
        return modelList;
    }

    private long calculateTotalSize(Map<String, List<File>> dataMap) {
        long size = 0L;
        for (String extension : dataMap.keySet()) {
            size += WorkspaceDataHelper.getSizeFor(dataMap.get(extension));
        }
        return size;
    }

    public int getTotalFiles() {
        return totalFileSize;
    }

    public long getTotalSize() {
        return totalByteSize;
    }
}
