/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import java.util.List;

import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StyledCellLabelProvider;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.stats.rcp.interfaces.ICategoryChildItem;
import org.eclipse.recommenders.stats.rcp.interfaces.ICategoryItem;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.TreeCategory;
import org.eclipse.recommenders.stats.rcp.model.TreeItem;
import org.eclipse.recommenders.stats.rcp.ui.ExtensionPointReader;
import org.eclipse.recommenders.stats.rcp.ui.util.TreeViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;

public class NavigationViewer {

    private TreeViewer viewer;
    private Composite parent;

    public NavigationViewer(Composite detailCmp) {
        parent = detailCmp;
    }

    public void createViewer() {
        final TreeColumnLayout layout = createTreeColumnLayout(parent);
        viewer = new TreeViewer(parent, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);
        viewer.getTree().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        viewer.getTree().setLinesVisible(false);
        viewer.getTree().setHeaderVisible(false);
        createTreeColumn("", viewer, 130, layout, 50);

        viewer.setContentProvider(new TreeContentProvider());
        viewer.setLabelProvider(new TreeViewerLabelProvider());
        viewer.setSorter(new TreeViewerSorter());

        List<ICategoryItem> modelList = ExtensionPointReader.getInstance().createTreeObjectsFromExtensions();
        viewer.setInput(modelList);
        viewer.expandAll();
        viewer.setSelection(new StructuredSelection(viewer.getExpandedElements()[0]));
    }

    public void addSelectionChangedListener(final StackLayout stack, final Composite detailCmp) {
        viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            @Override
            public void selectionChanged(SelectionChangedEvent event) {
                ISelection selection = event.getSelection();
                if (selection.isEmpty()) {
                    return;
                }
                if (selection instanceof IStructuredSelection && selection instanceof TreeSelection) {
                    TreeSelection treeSelection = (TreeSelection) selection;
                    stack.topControl = getCompositeFromTreeItems(treeSelection.getFirstElement(), detailCmp);
                    detailCmp.layout();
                }
            }
        });
    }

    private Composite getCompositeFromTreeItems(final Object selection, final Composite detailCmp) {
        if (selection instanceof TreeCategory) {
            String description = ((TreeCategory) selection).getDescription();
            return new CategoryView(description).createContent(detailCmp);
        } else if (selection instanceof TreeItem) {
            IPageContent pageObject = ((TreeItem) selection).getPageObject();
            return pageObject.createContent(detailCmp);
        }
        // backup composite if nothing found (avoid NPE)
        return new Composite(detailCmp, SWT.NONE);
    }

    private TreeViewerColumn createTreeColumn(String header, TreeViewer viewer, int width, TreeColumnLayout layout,
            int weight) {
        final TreeViewerColumn column = new TreeViewerColumn(viewer, SWT.FILL);
        column.getColumn().setMoveable(true);
        column.getColumn().setText(header);
        column.getColumn().setToolTipText(header);
        column.getColumn().setAlignment(SWT.CENTER);
        column.getColumn().setWidth(width);
        column.getColumn().setResizable(true);
        layout.setColumnData(column.getColumn(), new ColumnWeightData(weight));
        return column;
    }

    private TreeColumnLayout createTreeColumnLayout(final Composite comp) {
        final TreeColumnLayout layout = new TreeColumnLayout();
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));
        comp.setLayout(layout);
        return layout;
    }

    public class TreeContentProvider implements ITreeContentProvider {
        private List<ICategoryItem> catList;

        @Override
        public void dispose() {
        }

        @SuppressWarnings("unchecked")
        @Override
        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
            catList = (List<ICategoryItem>) newInput;
        }

        @Override
        public Object[] getElements(Object inputElement) {
            return catList.toArray();
        }

        @Override
        public Object[] getChildren(Object parentElement) {
            if (parentElement instanceof ICategoryItem) {
                ICategoryItem cat = (ICategoryItem) parentElement;
                return cat.getItems().toArray();
            }
            return null;
        }

        @Override
        public Object getParent(Object element) {
            return null;
        }

        @Override
        public boolean hasChildren(Object element) {
            if (element instanceof ICategoryItem) {
                return true;
            }
            return false;
        }
    }

    public class TreeViewerLabelProvider extends StyledCellLabelProvider {
        @Override
        public void update(ViewerCell cell) {
            if (cell.getElement() instanceof ICategoryItem) {
                ICategoryItem category = (ICategoryItem) cell.getElement();
                cell.setImage(category.getIcon());
            } else if (cell.getElement() instanceof ICategoryChildItem) {
                ICategoryChildItem item = (ICategoryChildItem) cell.getElement();
                cell.setImage(item.getIcon());
            }
            cell.setText(cell.getElement().toString());
        }
    }
}
