/**
 * Copyright (c) 2013 Timur Achmetow.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Timur Achmetow - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;
import static org.eclipse.recommenders.utils.Checks.cast;

import java.util.Collection;

import javax.inject.Inject;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.recommenders.rcp.utils.TableSortConfigurator;
import org.eclipse.recommenders.stats.rcp.events.CompletionEvent;
import org.eclipse.recommenders.stats.rcp.interfaces.IDataPersistenceService;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.ui.data.StatisticData;
import org.eclipse.recommenders.stats.rcp.ui.util.TableSorters.ReceiverType;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.recommenders.utils.Bags;
import org.eclipse.recommenders.utils.names.ITypeName;
import org.eclipse.recommenders.utils.names.Names;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multiset.Entry;

public class ReceiverTypeView implements IPageContent {

    private final Collection<CompletionEvent> okayEvents;

    @Inject
    public ReceiverTypeView(IDataPersistenceService persistenceService) {
        okayEvents = StatisticData.create(persistenceService).getOkayEvents();
    }

    @Override
    public Composite createContent(Composite detailParent) {
        Multiset<ITypeName> b = HashMultiset.create();
        for (CompletionEvent e : okayEvents) {
            if (e.receiverType == null) {
                continue;
            }
            b.add(e.receiverType);
        }

        Composite newComp = createWrapperComposite(detailParent);
        new Label(newComp, SWT.NONE)
                .setText("Code completion was triggered most frequently on variables of these types:");

        Composite comp = new Composite(newComp, SWT.NONE);
        TableColumnLayout layout = createTableColumnLayout(comp);

        final TableViewer viewer = createTableViewer(comp);
        TableViewerColumn typeColumn = createColumn("Type", viewer, 450, layout, 60, SWT.LEFT);
        TableViewerColumn countColumn = createColumn("Count", viewer, 100, layout, 30, SWT.RIGHT);

        TableSortConfigurator.newConfigurator(viewer, new Action() {
            @Override
            public void run() {
                viewer.refresh();
            };
        }).add(typeColumn.getColumn(), ReceiverType.TYPE_COMPARATOR)
                .add(countColumn.getColumn(), ReceiverType.COUNT_COMPARATOR)
                .initialize(countColumn.getColumn(), SWT.DOWN).configure();

        TableViewerFactory.addMenu(viewer);
        viewer.setContentProvider(new ArrayContentProvider());
        viewer.setLabelProvider(new TypeNameLabelProvider());
        viewer.setInput(Bags.topUsingCount(b, 30));
        return newComp;
    }

    public class TypeNameLabelProvider extends CellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            String cellText = null;
            Entry<ITypeName> e = cast(cell.getElement());

            switch (cell.getColumnIndex()) {
            case 0:
                cellText = Names.vm2srcQualifiedType(e.getElement());
                break;
            case 1:
                cellText = Integer.toString(e.getCount());
                break;
            }

            if (cellText != null) {
                cell.setText(cellText);
            }
        }
    }
}
