/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;

import java.util.Comparator;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.viewers.deferred.DeferredContentProvider;
import org.eclipse.jface.viewers.deferred.SetModel;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceProjectType;
import org.eclipse.recommenders.stats.rcp.ui.data.WorkspaceProjectData;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

public class WorkspaceProjectView implements IPageContent {

    @Override
    public Composite createContent(Composite detailParent) {
        final Composite parent = TableViewerFactory.createWrapperComposite(detailParent);

        final Label projectNumberLabel = new Label(parent, SWT.NONE);
        final Label projectSizeLabel = new Label(parent, SWT.NONE);
        final Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));

        final TableColumnLayout layout = createTableColumnLayout(comp);
        final TableViewer viewer = createVirtualTableViewer(comp);

        createColumn("Name", viewer, 80, layout, 30, SWT.CENTER);
        createColumn("Type", viewer, 60, layout, 15, SWT.CENTER);
        createColumn("Files", viewer, 30, layout, 10, SWT.CENTER);
        TableViewerColumn sizeCol = createColumn("Size", viewer, 30, layout, 10, SWT.CENTER);
        viewer.getTable().setSortDirection(SWT.DOWN);
        viewer.getTable().setSortColumn(sizeCol.getColumn());
        TableViewerFactory.addMenu(viewer);

        viewer.setLabelProvider(new WorkspaceProjectLabelProvider());
        viewer.setContentProvider(new DeferredContentProvider(createComparator(viewer)));

        SetModel times = new SetModel();
        viewer.setInput(times);
        startProjectCollectorJob(times, projectNumberLabel, projectSizeLabel);

        return parent;
    }

    private Comparator<WorkspaceProjectType> createComparator(final TableViewer viewer) {
        return new Comparator<WorkspaceProjectType>() {
            @Override
            public int compare(WorkspaceProjectType o1, WorkspaceProjectType o2) {
                Long bytes1 = Long.valueOf(o1.getBytes());
                Long bytes2 = Long.valueOf(o2.getBytes());
                return Long.compare(bytes2, bytes1);
            }
        };
    }

    private void startProjectCollectorJob(final SetModel times, final Label projectNumberLabel,
            final Label projectSizeLabel) {
        final Job findFilesJob = new Job("Collect all Projects") {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask("Collect Projects from Workspace", IProgressMonitor.UNKNOWN);

                WorkspaceProjectData workspaceData = new WorkspaceProjectData();
                times.addAll(workspaceData.readAllProjects());
                updateUI(projectNumberLabel, projectSizeLabel, workspaceData);

                if (monitor.isCanceled()) {
                    return Status.CANCEL_STATUS;
                }
                return Status.OK_STATUS;
            }

            private void updateUI(final Label projectNumberLabel, final Label projectSizeLabel,
                    final WorkspaceProjectData workspaceData) {
                Display.getDefault().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        String fileSize = FileUtils.byteCountToDisplaySize(workspaceData.getTotalSize());
                        projectNumberLabel.setText("Total Number of all Projects: " + workspaceData.getTotalProjects());
                        projectSizeLabel.setText("File Size of these Projects: " + fileSize);
                        projectNumberLabel.pack();
                        projectSizeLabel.pack();
                    }
                });
            }
        };
        findFilesJob.setPriority(Job.SHORT);
        findFilesJob.schedule();
    }

    private class WorkspaceProjectLabelProvider extends CellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            String cellText = null;
            WorkspaceProjectType workspaceProject = (WorkspaceProjectType) cell.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText = workspaceProject.getName();
                break;
            case 1:
                cellText = workspaceProject.getType();
                break;
            case 2:
                cellText = workspaceProject.getFileCount();
                break;
            case 3:
                cellText = FileUtils.byteCountToDisplaySize(workspaceProject.getBytes());
                break;
            }

            if (cellText != null) {
                cell.setText(cellText);
            }
        }
    }
}
