/**
 * Copyright (c) 2015 Akif Etkue.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Akif Etkue - Initial API and implementation
 */
package org.eclipse.recommenders.stats.rcp.ui.views;

import static org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory.*;

import java.util.Comparator;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.jface.viewers.deferred.DeferredContentProvider;
import org.eclipse.jface.viewers.deferred.SetModel;
import org.eclipse.recommenders.stats.rcp.interfaces.IPageContent;
import org.eclipse.recommenders.stats.rcp.model.WorkspaceContentType;
import org.eclipse.recommenders.stats.rcp.ui.data.WorkspaceFileData;
import org.eclipse.recommenders.stats.rcp.ui.util.TableViewerFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

public class WorkspaceView implements IPageContent {

    @Override
    public Composite createContent(Composite detailParent) {
        final Composite parent = TableViewerFactory.createWrapperComposite(detailParent);

        final Label fileNumberLabel = new Label(parent, SWT.NONE);
        final Label fileSizeLabel = new Label(parent, SWT.NONE);
        final Composite comp = new Composite(parent, SWT.NONE);
        comp.setLayout(new GridLayout());
        comp.setLayoutData(new GridData(GridData.FILL_BOTH));

        final TableColumnLayout layout = createTableColumnLayout(comp);
        final TableViewer viewer = createVirtualTableViewer(comp);

        createColumn("File Type", viewer, 60, layout, 10, SWT.CENTER);
        TableViewerColumn typeCol = createColumn("Count", viewer, 60, layout, 10, SWT.CENTER);
        createColumn("Size", viewer, 60, layout, 10, SWT.CENTER);
        viewer.getTable().setSortDirection(SWT.DOWN);
        viewer.getTable().setSortColumn(typeCol.getColumn());
        TableViewerFactory.addMenu(viewer);

        viewer.setLabelProvider(new WorkspaceFileLabelProvider());
        viewer.setContentProvider(new DeferredContentProvider(createComparator()));

        SetModel times = new SetModel();
        viewer.setInput(times);
        startFileCollectorJob(fileNumberLabel, fileSizeLabel, times);

        return parent;
    }

    private Comparator<WorkspaceContentType> createComparator() {
        return new Comparator<WorkspaceContentType>() {
            @Override
            public int compare(WorkspaceContentType o1, WorkspaceContentType o2) {
                Integer count1 = Integer.valueOf(o1.getCount());
                Integer count2 = Integer.valueOf(o2.getCount());
                return Integer.compare(count2, count1);
            }
        };
    }

    private void startFileCollectorJob(final Label fileNumberLabel, final Label fileSizeLabel, final SetModel times) {
        final Job findFilesJob = new Job("Collect all Files") {
            @Override
            protected IStatus run(IProgressMonitor monitor) {
                monitor.beginTask("Collect Files from Workspace", IProgressMonitor.UNKNOWN);

                WorkspaceFileData workspaceData = new WorkspaceFileData();
                times.addAll(workspaceData.getPreparedData());

                updateUI(fileNumberLabel, fileSizeLabel, workspaceData);

                if (monitor.isCanceled()) {
                    return Status.CANCEL_STATUS;
                }
                return Status.OK_STATUS;
            }

            private void updateUI(final Label fileNumberLabel, final Label fileSizeLabel,
                    final WorkspaceFileData workspaceData) {
                Display.getDefault().asyncExec(new Runnable() {
                    @Override
                    public void run() {
                        String fileSize = FileUtils.byteCountToDisplaySize(workspaceData.getTotalSize());
                        fileNumberLabel.setText("Total Number of all Files: " + workspaceData.getTotalFiles());
                        fileSizeLabel.setText("File Size of these Files: " + fileSize);
                        fileNumberLabel.pack();
                        fileSizeLabel.pack();
                    }
                });
            }
        };
        findFilesJob.setPriority(Job.SHORT);
        findFilesJob.schedule();
    }

    private class WorkspaceFileLabelProvider extends CellLabelProvider {

        @Override
        public void update(ViewerCell cell) {
            String cellText = null;
            WorkspaceContentType workspaceFile = (WorkspaceContentType) cell.getElement();

            switch (cell.getColumnIndex()) {
            case 0:
                cellText = workspaceFile.getLabel();
                break;
            case 1:
                cellText = workspaceFile.getCount();
                break;
            case 2:
                cellText = FileUtils.byteCountToDisplaySize(workspaceFile.getBytes());
                break;
            }

            if (cellText != null) {
                cell.setText(cellText);
            }
        }
    }
}
