/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.recommenders.snipmatch;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.analysis.PerFieldAnalyzerWrapper;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.DefaultSimilarity;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Similarity;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.Version;
import org.eclipse.recommenders.snipmatch.ISnippet;
import org.eclipse.recommenders.snipmatch.ISnippetRepository;
import org.eclipse.recommenders.snipmatch.MultiFieldPrefixQueryParser;
import org.eclipse.recommenders.snipmatch.Snippet;
import org.eclipse.recommenders.utils.IOUtils;
import org.eclipse.recommenders.utils.Recommendation;
import org.eclipse.recommenders.utils.Urls;
import org.eclipse.recommenders.utils.gson.GsonUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSnippetRepository
implements ISnippetRepository {
    private static final int MAX_SEARCH_RESULTS = 100;
    private static final int CACHE_SIZE = 200;
    private static final Set<String> EMPTY_STOPWORDS = Collections.emptySet();
    private static final String F_NAME = "name";
    private static final String F_DESCRIPTION = "description";
    private static final String F_EXTRA_SEARCH_TERM = "extra";
    private static final String F_TAG = "tag";
    private static final String F_PATH = "path";
    private static final String F_UUID = "uuid";
    private static final float NAME_BOOST = 4.0f;
    private static final float DESCRIPTION_BOOST = 2.0f;
    private static final float EXTRA_SEARCH_TERM_BOOST = 2.0f;
    private static final float TAG_BOOST = 1.0f;
    private Logger log = LoggerFactory.getLogger(this.getClass());
    private volatile int timesOpened = 0;
    private final Lock readLock;
    private final Lock writeLock;
    private final File snippetsdir;
    private final File indexdir;
    private final String repoUrl;
    private Directory directory;
    private IndexReader reader;
    private final Analyzer analyzer;
    private final QueryParser parser;
    private final Similarity similarity;
    private final LoadingCache<File, Snippet> snippetCache = CacheBuilder.newBuilder().maximumSize(200L).build((CacheLoader)new CacheLoader<File, Snippet>(){

        public Snippet load(File file) throws Exception {
            Snippet snippet = (Snippet)GsonUtil.deserialize((File)file, Snippet.class);
            return snippet;
        }
    });

    public FileSnippetRepository(File basedir) {
        Preconditions.checkArgument((boolean)true, (Object)"The cache size needs to be larger than the maximum number of search results.");
        this.snippetsdir = new File(basedir, "snippets");
        this.indexdir = new File(basedir, "index");
        this.repoUrl = Urls.mangle((String)basedir.getAbsolutePath());
        this.analyzer = this.createAnalyzer();
        this.parser = this.createParser();
        this.similarity = new IgnoreDocFrequencySimilarity();
        ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();
        this.readLock = readWriteLock.readLock();
        this.writeLock = readWriteLock.writeLock();
    }

    private Analyzer createAnalyzer() {
        StandardAnalyzer standardAnalyzer = new StandardAnalyzer(Version.LUCENE_35, EMPTY_STOPWORDS);
        HashMap analyzers = Maps.newHashMap();
        analyzers.put(F_NAME, standardAnalyzer);
        analyzers.put(F_DESCRIPTION, standardAnalyzer);
        analyzers.put(F_EXTRA_SEARCH_TERM, standardAnalyzer);
        analyzers.put(F_TAG, standardAnalyzer);
        analyzers.put(F_UUID, new KeywordAnalyzer());
        return new PerFieldAnalyzerWrapper((Analyzer)new KeywordAnalyzer(), (Map)analyzers);
    }

    private QueryParser createParser() {
        String[] searchFields = new String[]{F_NAME, F_DESCRIPTION, F_EXTRA_SEARCH_TERM, F_TAG};
        ImmutableMap boosts = ImmutableMap.of((Object)F_NAME, (Object)Float.valueOf(4.0f), (Object)F_DESCRIPTION, (Object)Float.valueOf(2.0f), (Object)F_EXTRA_SEARCH_TERM, (Object)Float.valueOf(2.0f), (Object)F_TAG, (Object)Float.valueOf(1.0f));
        MultiFieldPrefixQueryParser parser = new MultiFieldPrefixQueryParser(Version.LUCENE_35, searchFields, this.analyzer, (Map<String, Float>)boosts, F_NAME, F_DESCRIPTION, F_EXTRA_SEARCH_TERM);
        parser.setDefaultOperator(QueryParser.Operator.AND);
        return parser;
    }

    public void open() throws IOException {
        this.writeLock.lock();
        try {
            ++this.timesOpened;
            if (this.timesOpened > 1) {
                return;
            }
            this.snippetsdir.mkdirs();
            this.indexdir.mkdirs();
            this.directory = FSDirectory.open((File)this.indexdir);
            this.index();
            this.reader = IndexReader.open((Directory)this.directory);
        }
        finally {
            this.writeLock.unlock();
        }
    }

    public void index() throws IOException {
        this.writeLock.lock();
        try {
            File[] snippetFiles = this.snippetsdir.listFiles((FileFilter)new SuffixFileFilter(".json"));
            this.doIndex(snippetFiles);
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void doIndex(File[] snippetFiles) throws IOException {
        IndexWriterConfig config = new IndexWriterConfig(Version.LUCENE_35, this.analyzer);
        config.setOpenMode(IndexWriterConfig.OpenMode.CREATE);
        IndexWriter writer = new IndexWriter(this.directory, config);
        try {
            this.snippetCache.invalidateAll();
            File[] fileArray = snippetFiles;
            int n = snippetFiles.length;
            int n2 = 0;
            while (n2 < n) {
                File snippetFile = fileArray[n2];
                try {
                    ISnippet snippet = (ISnippet)this.snippetCache.get((Object)snippetFile);
                    String path = snippetFile.getPath();
                    this.indexSnippet(writer, snippet, path);
                }
                catch (Exception e) {
                    this.log.error("Failed to index snippet in " + snippetFile, (Throwable)e);
                }
                ++n2;
            }
        }
        finally {
            writer.close();
        }
        if (this.reader != null) {
            this.reader = IndexReader.openIfChanged((IndexReader)this.reader);
        }
    }

    private void indexSnippet(IndexWriter writer, ISnippet snippet, String path) throws CorruptIndexException, IOException {
        Document doc = new Document();
        doc.add((Fieldable)new Field(F_PATH, path, Field.Store.YES, Field.Index.NO));
        doc.add((Fieldable)new Field(F_UUID, snippet.getUuid().toString(), Field.Store.NO, Field.Index.NOT_ANALYZED));
        String name = snippet.getName();
        doc.add((Fieldable)new Field(F_NAME, name, Field.Store.YES, Field.Index.ANALYZED));
        String description = snippet.getDescription();
        doc.add((Fieldable)new Field(F_DESCRIPTION, description, Field.Store.YES, Field.Index.ANALYZED));
        for (String tag : snippet.getTags()) {
            doc.add((Fieldable)new Field(F_TAG, tag, Field.Store.YES, Field.Index.ANALYZED_NO_NORMS));
        }
        for (String extraSearchTerm : snippet.getExtraSearchTerms()) {
            doc.add((Fieldable)new Field(F_EXTRA_SEARCH_TERM, extraSearchTerm, Field.Store.YES, Field.Index.ANALYZED));
        }
        writer.addDocument(doc);
    }

    @VisibleForTesting
    public boolean isOpen() {
        return this.timesOpened > 0;
    }

    private ImmutableSet<Recommendation<ISnippet>> getSnippets() {
        this.readLock.lock();
        try {
            Preconditions.checkState((boolean)this.isOpen());
            HashSet res = Sets.newHashSet();
            File[] fileArray = this.snippetsdir.listFiles((FileFilter)new SuffixFileFilter(".json"));
            int n = fileArray.length;
            int n2 = 0;
            while (n2 < n) {
                File fSnippet = fileArray[n2];
                try {
                    ISnippet snippet = (ISnippet)this.snippetCache.get((Object)fSnippet);
                    res.add(Recommendation.newRecommendation((Object)snippet, (double)0.0));
                }
                catch (Exception e) {
                    this.log.error("Error while loading snippet from file {}", (Object)fSnippet.getAbsolutePath(), (Object)e);
                }
                ++n2;
            }
            ImmutableSet immutableSet = ImmutableSet.copyOf((Collection)res);
            return immutableSet;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @Override
    public List<Recommendation<ISnippet>> search(String query) {
        if (StringUtils.isBlank((CharSequence)query)) {
            return ImmutableList.copyOf(this.getSnippets());
        }
        return this.doSearch(query, Integer.MAX_VALUE);
    }

    @Override
    public List<Recommendation<ISnippet>> search(String query, int maxResults) {
        if (StringUtils.isBlank((CharSequence)query)) {
            return Collections.emptyList();
        }
        return this.doSearch(query, Math.min(maxResults, 100));
    }

    private List<Recommendation<ISnippet>> doSearch(String query, int maxResults) {
        this.readLock.lock();
        try {
            Preconditions.checkState((boolean)this.isOpen());
            LinkedList results = Lists.newLinkedList();
            try {
                Map<File, Float> snippetFiles = this.searchSnippetFiles(query, maxResults);
                for (Map.Entry<File, Float> entry : snippetFiles.entrySet()) {
                    ISnippet snippet = (ISnippet)this.snippetCache.get((Object)entry.getKey());
                    results.add(Recommendation.newRecommendation((Object)snippet, (double)entry.getValue().floatValue()));
                }
            }
            catch (Exception e) {
                this.log.error("Exception occurred while searching the snippet index.", (Throwable)e);
            }
            LinkedList linkedList = results;
            return linkedList;
        }
        finally {
            this.readLock.unlock();
        }
    }

    /*
     * Loose catch block
     */
    private Map<File, Float> searchSnippetFiles(String query, int maxResults) {
        LinkedHashMap results;
        block9: {
            Map<File, Float> map;
            results = Maps.newLinkedHashMap();
            IndexSearcher searcher = null;
            try {
                Query q = this.parser.parse(query);
                searcher = new IndexSearcher(this.reader);
                searcher.setSimilarity(this.similarity);
                float maxScore = 0.0f;
                ScoreDoc[] scoreDocArray = searcher.search((Query)q, null, (int)maxResults).scoreDocs;
                int n = searcher.search((Query)q, null, (int)maxResults).scoreDocs.length;
                int n2 = 0;
                while (n2 < n) {
                    ScoreDoc hit = scoreDocArray[n2];
                    Document doc = searcher.doc(hit.doc);
                    results.put(new File(doc.get(F_PATH)), Float.valueOf(hit.score));
                    if (hit.score > maxScore) {
                        maxScore = hit.score;
                    }
                    ++n2;
                }
                map = this.normalizeValues(results, maxScore);
                IOUtils.closeQuietly((Closeable)searcher);
            }
            catch (ParseException e) {
                this.log.error("Failed to parse query", (Throwable)e);
                break block9;
            }
            catch (Exception e2) {
                this.log.error("Exception occurred while searching the snippet index.", (Throwable)e2);
                break block9;
                {
                    catch (Throwable throwable) {
                        throw throwable;
                    }
                }
            }
            finally {
                IOUtils.closeQuietly(searcher);
            }
            return map;
        }
        return results;
    }

    private Map<File, Float> normalizeValues(Map<File, Float> results, final float maxScore) {
        return Maps.transformValues(results, (Function)new Function<Float, Float>(){

            public Float apply(Float input) {
                return Float.valueOf(input.floatValue() / maxScore);
            }
        });
    }

    @Override
    public boolean hasSnippet(UUID uuid) {
        this.readLock.lock();
        try {
            Preconditions.checkState((boolean)this.isOpen());
            boolean bl = !this.searchSnippetFiles("uuid:" + uuid, Integer.MAX_VALUE).isEmpty();
            return bl;
        }
        finally {
            this.readLock.unlock();
        }
    }

    @Override
    public boolean delete(UUID uuid) throws IOException {
        this.writeLock.lock();
        try {
            Preconditions.checkState((boolean)this.isOpen());
            Map<File, Float> snippetFiles = this.searchSnippetFiles("uuid:" + uuid, Integer.MAX_VALUE);
            if (snippetFiles.isEmpty()) {
                return false;
            }
            ((File)Iterables.getOnlyElement(snippetFiles.keySet())).delete();
            this.index();
            return true;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @Override
    public boolean isDeleteSupported() {
        return true;
    }

    @Override
    public String getRepositoryLocation() {
        return this.repoUrl;
    }

    @Override
    public void close() {
        this.writeLock.lock();
        try {
            if (this.timesOpened == 0) {
                return;
            }
            if (this.timesOpened > 1) {
                --this.timesOpened;
                return;
            }
            if (this.timesOpened == 1) {
                this.timesOpened = 0;
                IOUtils.closeQuietly((Closeable)this.reader);
                IOUtils.closeQuietly((Closeable)this.directory);
                this.reader = null;
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @Override
    public void importSnippet(ISnippet snippet) throws IOException {
        this.writeLock.lock();
        try {
            Preconditions.checkState((boolean)this.isOpen());
            Snippet importSnippet = this.checkTypeAndConvertSnippet(snippet);
            Map<File, Float> snippetFiles = this.searchSnippetFiles("uuid:" + importSnippet.getUuid(), Integer.MAX_VALUE);
            File file = snippetFiles.isEmpty() ? new File(this.snippetsdir, importSnippet.getUuid() + ".json") : (File)Iterables.getOnlyElement(snippetFiles.keySet());
            FileWriter writer = new FileWriter(file);
            writer.write(GsonUtil.serialize((Object)importSnippet));
            writer.flush();
            writer.close();
            this.index();
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @Override
    public boolean isImportSupported() {
        return true;
    }

    private Snippet checkTypeAndConvertSnippet(ISnippet snippet) {
        if (snippet instanceof Snippet) {
            return (Snippet)snippet;
        }
        return Snippet.copy(snippet);
    }

    private static class IgnoreDocFrequencySimilarity
    extends DefaultSimilarity {
        private static final long serialVersionUID = 6048878092975074153L;

        private IgnoreDocFrequencySimilarity() {
        }

        public float tf(float freq) {
            return 1.0f;
        }

        public float idf(int docFreq, int numDocs) {
            return 1.0f;
        }
    }
}

