/*******************************************************************************
 * Copyright (c) 2008 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hugues Dubourg (Mia-Software) - initial API and implementation
 *    Gabriel Barbier (Mia-Software) - initial API and implementation
 *******************************************************************************/

package org.eclipse.gmt.modisco.kdm.uml2converter;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.Tools;
import org.eclipse.gmt.modisco.infra.common.core.internal.utils.UriUtils;
import org.eclipse.gmt.modisco.infra.common.core.logging.MoDiscoLogger;
import org.eclipse.m2m.atl.common.ATLLaunchConstants;
import org.eclipse.m2m.atl.core.ATLCoreException;
import org.eclipse.m2m.atl.core.IExtractor;
import org.eclipse.m2m.atl.core.IInjector;
import org.eclipse.m2m.atl.core.IModel;
import org.eclipse.m2m.atl.core.IReferenceModel;
import org.eclipse.m2m.atl.core.ModelFactory;
import org.eclipse.m2m.atl.core.emf.EMFModel;
import org.eclipse.m2m.atl.core.launch.ILauncher;
import org.eclipse.m2m.atl.core.service.CoreService;
import org.eclipse.m2m.atl.core.ui.vm.asm.ASMModelWrapper;
import org.eclipse.m2m.atl.drivers.emf4atl.ASMEMFModel;
import org.eclipse.m2m.atl.engine.parser.AtlParser;
import org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

public class KDMtoUML2Converter implements KdmToUmlConverterInterface {

	/**
	 * @deprecated replaced by MM_LOCATION (since 0.8.0M7)
	 */
	@Deprecated
	public final String mmLocation = "resources/metamodels"; //$NON-NLS-1$
	/**
	 * @deprecated replaced by TRANSFO_LOCATION (since 0.8.0M7)
	 */
	@Deprecated
	public final String transfoLocation = "resources/transformations"; //$NON-NLS-1$
	/**
	 * @deprecated replaced by KDM_MM_URI (since 0.8.0M7)
	 */
	@Deprecated
	public final String kdmMMUri = "http://www.eclipse.org/MoDisco/kdm/action"; //$NON-NLS-1$
	/**
	 * @deprecated replaced by UML_MM_URI (since 0.8.0M7)
	 */
	@Deprecated
	public final String umlMMUri = "http://www.eclipse.org/uml2/2.1.0/UML"; //$NON-NLS-1$

	public static final String MM_LOCATION = "resources/metamodels"; //$NON-NLS-1$
	public static final String TRANSFO_LOCATION = "resources/transformations"; //$NON-NLS-1$
	public static final String KDM_MM_URI = "http://www.eclipse.org/MoDisco/kdm/action"; //$NON-NLS-1$
	public static final String UML_MM_URI = "http://www.eclipse.org/uml2/2.1.0/UML"; //$NON-NLS-1$
	private static final String ATL_MM_PATH = "http://www.eclipse.org/gmt/2005/ATL"; //$NON-NLS-1$

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#getUML2ModelFromKDMModel(org.eclipse.emf.ecore.resource.Resource, boolean, org.eclipse.emf.common.util.URI)
	 */
	public Resource[] getUML2ModelFromKDMModel(final Resource kdmModelUri,
			final boolean generateTraces, final URI umlTargetModelUri) throws IOException, ATLCoreException {
		URL transformation;
		if (generateTraces) {
			transformation = this.getClass().getResource(
					KDMtoUML2Converter.TRANSFO_LOCATION
							+ "/KDMtoUMLWithTraces.asm"); //$NON-NLS-1$
		} else {
			transformation = this.getClass().getResource(
					KDMtoUML2Converter.TRANSFO_LOCATION + "/KDMtoUML.asm"); //$NON-NLS-1$
		}

		return this.getUML2ModelFromKDMModelWithCustomTransformation(
				kdmModelUri.getURI(), generateTraces, transformation,
				umlTargetModelUri);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#getUML2ModelFromKDMModelWithCustomTransformation(org.eclipse.emf.common.util.URI, boolean, java.net.URL, org.eclipse.emf.common.util.URI)
	 */
	public Resource[] getUML2ModelFromKDMModelWithCustomTransformation(
			final URI kdmSourceModelUri, final boolean generateTraces,
			final URL transformation, final URI umlTargetModelUri) throws IOException, ATLCoreException {

		Map<String, String> modelHandlers = new HashMap<String, String>();
		String umlMetaModelName = "uml"; //$NON-NLS-1$
		modelHandlers.put(umlMetaModelName, "UML2"); //$NON-NLS-1$
		String kdmMetaModelName = "kdm"; //$NON-NLS-1$
		modelHandlers.put(kdmMetaModelName, "EMF"); //$NON-NLS-1$
		String amwMetaModelName = "Trace"; //$NON-NLS-1$
		if (generateTraces) {
			modelHandlers.put(amwMetaModelName, "EMF"); //$NON-NLS-1$
//			modelHandlers.put(amwMetaModelName, "AMW");
		}
		
		Map<String, Object> launcherOptions = new HashMap<String, Object>();
		launcherOptions.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);

		String launcherName = ATLLaunchConstants.REGULAR_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(launcherOptions);
		
		ModelFactory emfFactory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());
		IInjector emfInjector = CoreService.getInjector(emfFactory.getDefaultInjectorName());
		IExtractor emfExtractor = CoreService.getExtractor(emfFactory.getDefaultExtractorName());
		
//		ModelFactory umlFactory = CoreService.createModelFactory("UML2");
		ModelFactory umlFactory = emfFactory;
		IInjector umlInjector = CoreService.getInjector(umlFactory.getDefaultInjectorName());
		IExtractor umlExtractor = CoreService.getExtractor(umlFactory.getDefaultExtractorName());

		// load metamodel KDM
		Map<String, Object> referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(kdmMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", kdmMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", KDMtoUML2Converter.KDM_MM_URI); //$NON-NLS-1$
		IReferenceModel kdmMM = emfFactory.newReferenceModel(referenceModelOptions);
		emfInjector.inject(kdmMM, KDMtoUML2Converter.KDM_MM_URI);
		// load model KDM
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "kdmInput"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(kdmSourceModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = emfFactory.newModel(kdmMM, modelOptions);
		emfInjector.inject(input, UriUtils.toString(kdmSourceModelUri));
		launcher.addInModel(input, inModelName, kdmMetaModelName);
		
		// load meta model UML
		referenceModelOptions = new HashMap<String, Object>();
		referenceModelOptions.put("modelHandlerName", modelHandlers.get(umlMetaModelName)); //$NON-NLS-1$
		referenceModelOptions.put("modelName", umlMetaModelName); //$NON-NLS-1$
		referenceModelOptions.put("path", KDMtoUML2Converter.UML_MM_URI); //$NON-NLS-1$
		IReferenceModel umlMM = umlFactory.newReferenceModel(referenceModelOptions);
		umlInjector.inject(umlMM, KDMtoUML2Converter.UML_MM_URI);
		// load model UML
		modelOptions = new HashMap<String, Object>();
		inModelName = "umlOutput"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", UriUtils.toString(umlTargetModelUri)); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = umlFactory.newModel(umlMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, umlMetaModelName);
		
		IModel traceInstance = null;
		if (generateTraces) {
//			ModelFactory amwFactory = CoreService.createModelFactory("AMW");
			ModelFactory amwFactory = emfFactory;
			IInjector amwInjector = CoreService.getInjector(amwFactory.getDefaultInjectorName());
			IExtractor amwExtractor = CoreService.getExtractor(amwFactory.getDefaultExtractorName());
			
			URL mmwMMURL = this.getClass().getResource(
					KDMtoUML2Converter.MM_LOCATION + "/mmw_traceability.ecore"); //$NON-NLS-1$

			// load meta model AMW
			referenceModelOptions = new HashMap<String, Object>();
			referenceModelOptions.put("modelHandlerName", modelHandlers.get(amwMetaModelName)); //$NON-NLS-1$
			referenceModelOptions.put("modelName", amwMetaModelName); //$NON-NLS-1$
			referenceModelOptions.put("path", mmwMMURL.toString()); //$NON-NLS-1$
			IReferenceModel amwMM = amwFactory.newReferenceModel(referenceModelOptions);
			amwInjector.inject(amwMM, mmwMMURL.openStream(), Collections.EMPTY_MAP);
			// load model AMW
			modelOptions = new HashMap<String, Object>();
			inModelName = "trace"; //$NON-NLS-1$
			modelOptions.put("modelName", inModelName); //$NON-NLS-1$
			modelOptions.put("path", UriUtils.toString(umlTargetModelUri.trimFileExtension().appendFileExtension("trace.amw"))); //$NON-NLS-1$ //$NON-NLS-2$
			modelOptions.put("newModel", true); //$NON-NLS-1$
			traceInstance = amwFactory.newModel(amwMM, modelOptions);
			launcher.addOutModel(traceInstance, inModelName, amwMetaModelName);
		}
		
		

		final Map<String, Object> options = new HashMap<String, Object>();
		options.put("continueAfterError", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		options.put("printExecutionTime", "true"); //$NON-NLS-1$ //$NON-NLS-2$

		Job transformationThread = new Job(Messages.KDMtoUML2Converter_22) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
		}
		
		/*
		 * In case of UML model, profiles and stereotypes applications 
		 * have been made only after save of model (essentially to avoid
		 * bad ordering of profiles and stereotypes applications) 
		 * 
		 */
		umlExtractor.extract(outputInstance, UriUtils.toString(umlTargetModelUri));
		if (generateTraces) {
			emfExtractor.extract(traceInstance, UriUtils.toString(umlTargetModelUri.trimFileExtension().appendFileExtension("trace.amw"))); //$NON-NLS-1$
		}

		Resource output = null;
		if (outputInstance instanceof ASMModelWrapper) {
			ASMModelWrapper wrapper = (ASMModelWrapper) outputInstance;
			ASMModel model = wrapper.getAsmModel();
			if (model instanceof ASMEMFModel) {
				output = ((ASMEMFModel) model).getExtent();
			}
		}
		Resource outputTrace = null;
		if (generateTraces) {
			if (traceInstance instanceof ASMModelWrapper) {
				ASMModelWrapper wrapper = (ASMModelWrapper) traceInstance;
				ASMModel model = wrapper.getAsmModel();
				if (model instanceof ASMEMFModel) {
					outputTrace = ((ASMEMFModel) model).getExtent();
				}
			}
		}

		return new Resource[] { output, outputTrace };
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#exportKdmToUmlTransformation(org.eclipse.core.runtime.IPath)
	 */
	public void exportKdmToUmlTransformation(final IPath pathParameter) {
		final InputStream transfoFileStream = this.getClass()
				.getResourceAsStream(
						KDMtoUML2Converter.TRANSFO_LOCATION + "/KDMtoUML.atl"); //$NON-NLS-1$
		IPath path = pathParameter;
		if (path.getFileExtension() == null
				|| !path.getFileExtension().equals("atl")) { //$NON-NLS-1$
			path = path.addFileExtension("atl"); //$NON-NLS-1$
		}
		final IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(
				path);
		final IProject destinationProject = file.getProject();
		Job job = new Job(Messages.KDMtoUML2Converter_27) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				try {
					/*
					 *  whatever the used method, we guaranty that
					 *  flow "transfoFileStream" will be closed
					 */
					if (file.exists()) {
						file.setContents(transfoFileStream, IResource.FORCE,
								monitor);
					} else {
						file.create(transfoFileStream, IResource.FORCE, monitor);
					}
				} catch (CoreException e) {
					return Status.CANCEL_STATUS;
				}
				return Status.OK_STATUS;
			}
		};
		// when copy job is done, open the file
		job.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(final IJobChangeEvent event) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						try {
							destinationProject.refreshLocal(
									IResource.DEPTH_INFINITE,
									new NullProgressMonitor());
							IWorkbenchPage page = PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getActivePage();
							IDE.openEditor(page, file);
						} catch (Exception e) {
							MoDiscoLogger.logError(e, KdmToUml2Activator
									.getDefault());
						}
					}
				});
			}
		});
		job.schedule();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#instrumentAtlTranformationWithTraceability(java.io.InputStream, org.eclipse.emf.common.util.URI)
	 */
	public IFile instrumentAtlTranformationWithTraceability(
			final InputStream sourceTransformation, final URI atlFileUri) throws ATLCoreException {
		URI sourceModelUri = atlFileUri.trimFileExtension().appendFragment("-ATL_source").appendFileExtension("ecore"); //$NON-NLS-1$ //$NON-NLS-2$
		URI targetModelUri = atlFileUri.trimFileExtension().appendFragment("-ATL_target").appendFileExtension("ecore"); //$NON-NLS-1$ //$NON-NLS-2$
		
		// transforms ATL file into ATL model
		IModel sourceModel = AtlParser.getDefault().parseToModel(sourceTransformation);
		if (sourceModel instanceof EMFModel) {
			EMFModel ecoreSourceModel = (EMFModel) sourceModel;
			Tools.saveModel(ecoreSourceModel.getResource(), sourceModelUri);
		}
		
		
		IModel targetModel = this.addTraceability(sourceModel, sourceModelUri, targetModelUri);
		
		if (targetModel instanceof EMFModel) {
			EMFModel ecoreTargetModel = (EMFModel) targetModel;
			Tools.saveModel(ecoreTargetModel.getResource(), targetModelUri);
		}
		IFile result = null;
		// transforms ATL model into ATL file
		try {
			result = this.parseFromModel(targetModel, atlFileUri);
		} catch (IOException e) {
			MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
		}

		return result;
	}

	private final IModel addTraceability(final IModel sourceModel, final URI sourceModelUri, final URI targetModelUri) throws ATLCoreException {
		final URL transformation = this.getClass().getResource(
				KDMtoUML2Converter.TRANSFO_LOCATION + "/ATL2WTracer.asm"); //$NON-NLS-1$

		Map<String, String> modelHandlers = new HashMap<String, String>();
		String atlMetaModelName = "ATL"; //$NON-NLS-1$
		modelHandlers.put(atlMetaModelName, "EMF"); //$NON-NLS-1$
		
		final Map<String, Object> options = new HashMap<String, Object>();
		options.put(ATLLaunchConstants.OPTION_MODEL_HANDLER, modelHandlers);
		options.put(ATLLaunchConstants.IS_REFINING, true);
		
		String launcherName = ATLLaunchConstants.EMF_VM_NAME;
		final ILauncher launcher = CoreService.getLauncher(launcherName);
		launcher.initialize(options);
		
		ModelFactory factory = CoreService.createModelFactory(launcher.getDefaultModelFactoryName());

		IInjector injector = CoreService.getInjector(factory.getDefaultInjectorName());
		IExtractor extractor = CoreService.getExtractor(factory.getDefaultExtractorName());
		
		// load meta model ATL
		IReferenceModel atlMM = factory.getBuiltInResource("ATL.ecore"); //$NON-NLS-1$
		// load model source
		Map<String, Object> modelOptions = new HashMap<String, Object>();
		String inModelName = "IN"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", sourceModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", false); //$NON-NLS-1$
		IModel input = factory.newModel(atlMM, modelOptions);
		injector.inject(input, sourceModelUri.toString());
		launcher.addInModel(input, inModelName, atlMetaModelName);
		// load model cible
		modelOptions = new HashMap<String, Object>();
		inModelName = "OUT"; //$NON-NLS-1$
		modelOptions.put("modelName", inModelName); //$NON-NLS-1$
		modelOptions.put("path", targetModelUri.toString()); //$NON-NLS-1$
		modelOptions.put("newModel", true); //$NON-NLS-1$
		IModel outputInstance = factory.newModel(atlMM, modelOptions);
		launcher.addOutModel(outputInstance, inModelName, atlMetaModelName);
		
		/*
		 * Encapsulate ATL transformation into a new Thread
		 * to reset the Stack (avoid stack overflow)
		 */
		options.put("continueAfterError", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		options.put("printExecutionTime", "true"); //$NON-NLS-1$ //$NON-NLS-2$

		Job transformationThread = new Job(Messages.KDMtoUML2Converter_42) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = Status.OK_STATUS;
				try {
					launcher.launch(ILauncher.RUN_MODE, monitor, options,
							transformation.openStream());
				} catch (IOException e) {
					result = Status.CANCEL_STATUS;
					MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
				}

				return result;
			}
		};
		transformationThread.schedule();
		try {
			transformationThread.join();
		} catch (InterruptedException e) {
			MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
		}
		
		extractor.extract(outputInstance, targetModelUri.toString());
		
		return outputInstance;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.gmt.modisco.kdm.uml2converter.KdmToUmlConverterInterface#parseFromModel(org.eclipse.m2m.atl.engine.vm.nativelib.ASMModel, org.eclipse.emf.common.util.URI)
	 */
	public final IFile parseFromModel(final IModel transformationModel,
			final URI atlFileUri) throws IOException {
		IFile file = null;
		OutputStream out = null;
		IPath atlFilePath = null;
		if (atlFileUri.isPlatformResource()) {
			atlFilePath = new Path(atlFileUri.toPlatformString(false));
			file = ResourcesPlugin.getWorkspace().getRoot().getFile(atlFilePath);
			PipedInputStream in = new PipedInputStream();
			out = new PipedOutputStream(in);
			try {
				AtlParser.getDefault().extract(transformationModel, out, null);
			} catch (Exception e) {
				MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
			} finally {
				try {
					out.close();
				} catch (IOException ioe) {
					MoDiscoLogger
							.logError(ioe, KdmToUml2Activator.getDefault());
				}
			}
			try {
				if (file.exists()) {
					file.setContents(in, IResource.FORCE, null);
				} else {
					file.create(in, IResource.FORCE, null);
				}
			} catch (CoreException e) {
				MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
			}
		} else {
			atlFilePath = new Path(atlFileUri.toFileString());
			out = new FileOutputStream(atlFilePath.toFile());
			try {
				AtlParser.getDefault().extract(transformationModel, out, null);
			} catch (Exception e) {
				MoDiscoLogger.logError(e, KdmToUml2Activator.getDefault());
			} finally {
				try {
					out.close();
				} catch (IOException ioe) {
					MoDiscoLogger
							.logError(ioe, KdmToUml2Activator.getDefault());
				}
			}
		}

		return file;
	}
}
