/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.workflow.ui;

import java.util.HashMap;
import java.util.Map.Entry;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.gmt.modisco.infra.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constants;
import org.eclipse.gmt.modisco.workflow.DriverInterface;
import org.eclipse.gmt.modisco.workflow.controller.ControllerDriverPage;
import org.eclipse.gmt.modisco.workflow.ui.common.Common;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

/**
 * This class defines a page of Wizard. This page allows to load a MoDiscoWork
 * in MoDiscoWorkflow from all launcher of Eclipse.
 * 
 * @author Nicolas Payneau
 */
public class LoadLauncherPage extends WizardPage {

	private Table launchersTable;
	private ControllerDriverPage controller;
	private HashMap<TableItem, ILaunchConfiguration> launchConfigurationMap;
	private Composite container;
	private TableItem[] itemsSelected;

	public TableItem[] getItemsSelected() {
		return this.itemsSelected;
	}

	/**
	 * Class constructor
	 * 
	 * @param pageName
	 *            Name of the wizard page
	 * @param controller
	 *            The controller managing the display
	 */
	protected LoadLauncherPage(final String pageName, final ControllerDriverPage controller) {
		super(pageName, pageName, null);
		this.itemsSelected = new TableItem[] {};
		this.setImageDescriptor(Activator
				.getImageDescriptor(Constants.pathBigImage));
		this.controller = controller;
		this.launchConfigurationMap = new HashMap<TableItem, ILaunchConfiguration>();
		this.setDescription(Messages.LoadLauncherPage_0);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets
	 * .Composite)
	 */
	public void createControl(final Composite parent) {
		this.container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		this.container.setLayout(layout);
		layout.numColumns = 3;
		layout.verticalSpacing = 9;

		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData.verticalAlignment = org.eclipse.swt.layout.GridData.FILL;
		gridData.grabExcessVerticalSpace = true;
		this.launchersTable = new Table(this.container, SWT.BORDER
				| SWT.V_SCROLL | SWT.MULTI);
		this.launchersTable.setLayoutData(gridData);
		this.launchersTable.addSelectionListener(new SelectionListener() {
			public void widgetDefaultSelected(final SelectionEvent e) {
			}

			public void widgetSelected(final SelectionEvent e) {
				LoadLauncherPage.this.itemsSelected = LoadLauncherPage.this.launchersTable
						.getSelection();
				setPageComplete(true);
			}
		});
		this.launchersTable.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseDoubleClick(final MouseEvent e) {
				LoadLauncherPage.this.getWizard().performFinish();
				LoadLauncherPage.this.getShell().close();
			}
		});
		loadLauncher();
		initializeDialogUnits(this.container);
		setControl(this.container);

	}

	/**
	 * Load all the launchers in the table which one driver is defined
	 */
	private void loadLauncher() {
		ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
		HashMap<String, DriverInterface> driverList = this.controller
				.getMapNameDriver();
		try {
			for (ILaunchConfiguration launch : manager
					.getLaunchConfigurations()) {
				for (Entry<String, DriverInterface> entry : driverList
						.entrySet()) {
					if (entry.getValue().getTypeLaunchConfig().equals(
							launch.getType().getIdentifier())) {
						TableItem item = new TableItem(this.launchersTable,
								SWT.NULL);
						item.setText(launch.getName());
						item.setImage(Common.getInstance().retrieveImageDriver(
								entry.getValue().getTypeLaunchConfig()));
						this.launchConfigurationMap.put(item, launch);
						break;
					}
				}
			}
		} catch (CoreException e) {
			MoDiscoLogger.logError(e, Activator.getDefault());
		}
		setMessage(Messages.LoadLauncherPage_1);
		setPageComplete(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	@Override
	public void setVisible(final boolean visible) {
		if (visible) {
			this.getShell().setSize(400, 400);
			this.getShell().setLocation(
					Common.getInstance().centerShell(Display.getDefault(),
							this.getShell()));
		}
		super.setVisible(visible);
	}

	// Getter
	public HashMap<TableItem, ILaunchConfiguration> getLaunchConfigurationMap() {
		return this.launchConfigurationMap;
	}
}
