/* Generic oprofile information class for opmxl.
   Written by Keith Seitz <keiths@redhat.com>
   Copyright 2003, Red Hat, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of Red Hat, Inc. nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.  */

#include "opinfo.h"

#include <sstream>
#include <string.h>

#include "xmlfmt.h"

using namespace std;

// From liboputil.a. Sadly this won't work if we want to enable
// remote-system profiling.
extern "C" double op_cpu_frequency (void);

// Forward declaration
static void __output_unit_mask_info (ostream& os, const opinfo::event_t* e);

// Constructor
opinfo::opinfo (op_cpu cpu_type, string dir)
  : _cpu_type (cpu_type), _dir (dir)
{
}

// Returns the number of counters for this cpu type
int
opinfo::get_nr_counters (void) const
{
  return op_get_nr_counters (_cpu_type);
}

// Returns (in LIST) a list of valid events for the given counter
void
opinfo::get_events (eventlist_t& list, int ctr) const
{
  struct list_head* events, *p;
  events = op_events (_cpu_type);

  list_for_each (p, events)
    {
      struct op_event* event = list_entry (p, struct op_event, event_next);
      if (/*event->counter_mask == CTR_ALL || */ event->counter_mask & (1 << ctr))
	list.push_back (event);
    }
}

// Returns cpu frequency
double
opinfo::get_cpu_frequency (void) const
{
  return op_cpu_frequency ();
}

// Checks whether the given CTR, EVENT, and MASK are valid
opinfo::eventcheck
opinfo::check (int ctr, int event, int mask) const
{
  eventcheck::result_t result =
    static_cast<eventcheck::result_t> (op_check_events (ctr, event, mask, _cpu_type));
  return eventcheck (result);
}

// Converts the given string into a corresponding op_cpu (CPU_NO_GOOD if invalid)
op_cpu
opinfo::str_to_op_cpu (const char* const cpu_str)
{
  int i;
  for (i = 0; i < MAX_CPU_TYPE; ++i)
    {
      if (strcmp (op_get_cpu_type_str ((op_cpu) i), cpu_str) == 0)
	return (op_cpu) i;
    }

  return CPU_NO_GOOD;
}

// This actually outputs a bunch of information
ostream&
operator<< (ostream& os, const opinfo& info)
{
  os << startt ("info");

  // Output out number of counters and defaults
  os << startt ("num-counters") << info.get_nr_counters () << endt
     << startt ("cpu-frequency") << info.get_cpu_frequency () << endt
     << startt ("defaults")
     << startt ("sample-dir") << opinfo::get_default_samples_dir () << endt
     << startt ("lock-file") << opinfo::get_default_lock_file () << endt
     << startt ("log-file") << opinfo::get_default_log_file () << endt
     << startt ("dump-status") << opinfo::get_default_dump_status () << endt
     << endt;

  if (info.get_cpu_type() == CPU_TIMER_INT)
    {
      os << startt ("timer-mode") << "true" << endt;

      //create a fake timer event and output as normal
      char UM_0_DESC[] = TIMER_EVENT_MASK_UM_0_DESCRIPTION;
      char TIMER_NAME[] = TIMER_EVENT_NAME;
      char TIMER_DESC[] = TIMER_EVENT_DESCRIPTION;

      struct op_unit_mask mask;
      mask.unit_type_mask = utm_mandatory;
      mask.default_mask = TIMER_EVENT_MASK_DEFAULT_VALUE;
      mask.um[0].value = TIMER_EVENT_MASK_UM_0_VALUE;
      mask.um[0].desc = UM_0_DESC;
      mask.num = TIMER_EVENT_MASK_UM_NUM;

      opinfo::event_t timer_event;
      timer_event.name = TIMER_NAME;
      timer_event.desc = TIMER_DESC;
      timer_event.val = TIMER_EVENT_NUMBER;
      timer_event.min_count = TIMER_EVENT_MIN_COUNT;
      timer_event.counter_mask = TIMER_EVENT_COUNTER_MASK;
      timer_event.unit = &mask;

      os << startt ("event-list") << attrt ("counter", "0");

      os << (&timer_event);

      os << endt;
    }
  else
    {
      os << startt ("timer-mode") << "false" << endt;

      // Output event list
      for (int ctr = 0; ctr < info.get_nr_counters (); ++ctr)
        {
          opinfo::eventlist_t events;

          ostringstream ctr_s;
          ctr_s << ctr;
          os << startt ("event-list") << attrt ("counter", ctr_s.str ());
          info.get_events (events, ctr);
          opinfo::eventlist_t::iterator i;
          for (i = events.begin (); i != events.end (); ++i)
            os << (*i);
          os << endt;
        }
    }

  return os << endt;
}

// Prints the given EVENT on the given stream
ostream&
operator<< (ostream& os, const opinfo::event_t* event)
{
  os << startt ("event")
     << startt ("name") << event->name << endt
     << startt ("description") << event->desc << endt
     << startt ("value") << static_cast<int> (event->val) << endt
     << startt ("minimum") << static_cast<int> (event->min_count) << endt;

  // ouput unit mask info
  __output_unit_mask_info (os, event);

  return os << endt;
}

// Prints the given eventcheck on the given stream
ostream&
operator<< (ostream& os, const opinfo::eventcheck& check)
{
  static const char* errors[3] = {"invalid-event", "invalid-um", "invalid-counter"};

  os << startt ("check-events");

  if (check.get_result () == OP_OK_EVENT)
    os << startt ("result") << "ok" << endt;
  else
    {
      for (unsigned int i = 0; i < sizeof (errors) / sizeof (errors[0]); ++i)
	{
	  if ((check.get_result () & (1 << i)) > 0)
	    os << startt ("result") << errors[i] << endt;
	}
    }

  return os << endt;
}

// Convenience function to output unit mask information
static void
__output_unit_mask_info (ostream& os, const opinfo::event_t* event)
{
  const struct op_unit_mask* umask = event->unit;
  const char* type;

  switch (umask->unit_type_mask)
    {
    case utm_exclusive:
      type = "exclusive";   break;
    case utm_bitmask:
      type = "bitmask";     break;
    case utm_mandatory:
    default:
      type = "mandatory";   break;
    }

  os << startt ("unit-mask")
     << startt ("type") << type << endt
     << startt ("default") << umask->default_mask << endt;

  for (u32 i = 0; i < umask->num; ++i)
    {
      os << startt ("mask")
	 << startt ("value") << umask->um[i].value << endt
	 << startt ("description") << umask->um[i].desc << endt
	 << endt;
    }

  os << endt;
}
