/**
 * Copyright (c) 2009 Research In Motion Limited.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Jon Dearden (Research In Motion) - Initial Version
 */
package org.eclipse.mtj.core.sdk;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.mtj.core.sdk.device.IDevice;
import org.eclipse.mtj.core.sdk.device.IManagedDevice;

/**
 * Abstract implementaion of an ISDK that provides a device list by routing the
 * call to an SDK provided by an ISDKProvider. This implementation also requires
 * that a ManagedSDK provide access to its ISDKProvider and allows for the
 * duplication of a device and the deletion of a duplicated device.
 * 
 * @since 1.1
 */
public abstract class ManagedSDK implements ISDK {

    /**
     * This method supports the legacy getDeviceList() for imported SDKs by
     * routing the call to getProvidedDeviceList().
     */
    public List<IDevice> getDeviceList() throws CoreException {

        List<IManagedDevice> autoInstalledDevices = getProvidedDeviceList();

        if (autoInstalledDevices != null)
            return new ArrayList<IDevice>(getProvidedDeviceList());

        return new ArrayList<IDevice>();
    }

    /**
     * Return a list of devices provided by this SDK.
     */
    protected abstract List<IManagedDevice> getProvidedDeviceList()
            throws CoreException;

    /**
     * Obtain a reference to the SDK provider.
     */
    public abstract ISDKProvider getSDKProvider();
    
    /**
     * Delete a duplicated device. For SDK providers, only devices that were
     * duplicated by the user may be deleted.
     * 
     * @param device The IManagedDevice to be deleted.
     */
    public abstract void deleteDuplicateDevice(IManagedDevice device);

    /**
     * Cause a managed device to be duplicated. Duplicated devices are stored by
     * the SDK provider and not by MTJ.
     * 
     * @param device The IManagedDevice to be duplicated.
     * @param newName The new name for the duplicate. The name is generated by
     *            MTJ and must be unique.
     * @return The duplicated IDevice.
     */
    public abstract IDevice duplicateDevice(IManagedDevice device, String newName);

}
