/*
 * Copyright (c) 2007-2008-2009-2010 Obeo
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Obeo - initial API and implementation
 */
package org.eclipse.stp.sca.semantic.sawsdl.dnd;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.emf.type.core.commands.SetValueCommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.SetRequest;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.stp.sca.AnyExtension;
import org.eclipse.stp.sca.Component;
import org.eclipse.stp.sca.ComponentReference;
import org.eclipse.stp.sca.ComponentService;
import org.eclipse.stp.sca.ScaPackage;
import org.eclipse.stp.sca.diagram.dnd.IDragDropHandler;
import org.eclipse.swt.dnd.DropTargetEvent;

import sawsdl.SAWSDL;
import sawsdl.SawsdlFactory;
import sawsdl.SawsdlPackage;

/**
 * SAWSDL can be D'n'Ded on a Component, a ComponentService or a
 * ComponentReference.
 * 
 * @author Maxime Porhel - Obeo
 * @contributor Stephane Drapeau - Obeo (Added D'n'D on ComponentReference and
 *              ComponentService)
 * 
 */
public class SAWSDLImplementationDD implements IDragDropHandler {

	public SAWSDLImplementationDD() {
		// Nada
	}

	public AbstractTransactionalCommand getAbstractTransactionalCommand(
			final IGraphicalEditPart host, final GraphicalEditPart editPart,
			final Request request) {

		// Dropped modelReference.
		final String modelReference = extractModelReference(request);
		// ModelElement of the targeted edit part.
		final EObject modelElement = getCurrentModelObject(editPart);

		if (modelReference == null || modelElement == null)
			return null;

		AbstractTransactionalCommand command = null;
		final SAWSDL sawsdl = getCurrentSAWSDL(modelElement);

		if (sawsdl != null) {
			final List<String> references = new ArrayList<String>();
			if (sawsdl.getModelReference() != null) {
				references.addAll(sawsdl.getModelReference());
			}
			if (!references.contains(modelReference)) {
				references.add(modelReference);
				command = new AbstractTransactionalCommand(host
						.getEditingDomain(), "DnD SAWSDL information", null) {//$NON-NLS-1$

					@Override
					protected CommandResult doExecuteWithResult(
							IProgressMonitor monitor, IAdaptable info)
							throws ExecutionException {

						SetRequest reqSet = new SetRequest(editPart
								.getEditingDomain(), sawsdl,
								SawsdlPackage.Literals.SAWSDL__MODEL_REFERENCE,
								references);

						SetValueCommand operation = new SetValueCommand(reqSet);
						editPart.getDiagramEditDomain()
								.getDiagramCommandStack().execute(
										new ICommandProxy(operation));
						return CommandResult.newOKCommandResult();
					}
				};
			}
		} else {
			command = getCreateSAWSDLTransactionalCommand(host, editPart,
					modelElement, modelReference);
		}

		return command;
	}

	/**
	 * Build a command to create a SAWSDL element and add it to a model element.
	 * 
	 * @param host
	 * @param editPart
	 * @param modelElement
	 * @param modelReference
	 * @return
	 */
	private AbstractTransactionalCommand getCreateSAWSDLTransactionalCommand(
			IGraphicalEditPart host, final GraphicalEditPart editPart,
			final EObject modelElement, final String modelReference) {
		return new AbstractTransactionalCommand(host.getEditingDomain(),
				"DnD SAWSDL information", null) {//$NON-NLS-1$
			@Override
			protected CommandResult doExecuteWithResult(
					IProgressMonitor monitor, IAdaptable info)
					throws ExecutionException {

				SAWSDL newMod = SawsdlFactory.eINSTANCE.createSAWSDL();

				final List<String> values = new ArrayList<String>();
				values.add(modelReference);

				newMod.setModelReference(values);
				SetRequest reqSet = new SetRequest(editPart.getEditingDomain(),
						modelElement, getAnyExtensionGroup(modelElement),
						FeatureMapUtil.createEntry(
								SawsdlPackage.Literals.DOCUMENT_ROOT__SAWSDL,
								newMod));

				SetValueCommand operation = new SetValueCommand(reqSet);
				editPart.getDiagramEditDomain().getDiagramCommandStack()
						.execute(new ICommandProxy(operation));
				return CommandResult.newOKCommandResult();
			}
		};
	}

	/**
	 * Retrieve the first SAWSDL element of an EObject.
	 * 
	 * @param eObject
	 * @return
	 */
	private SAWSDL getCurrentSAWSDL(EObject eObject) {
		SAWSDL result = null;
		Method getAnyextension = null;
		EList<AnyExtension> anyextensions = null;

		Class<?> eObjectClass = eObject.getClass();

		try {
			getAnyextension = eObjectClass.getMethod("getAnyextension"); //$NON-NLS-1$
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		}

		if (getAnyextension != null) {
			try {
				anyextensions = (EList<AnyExtension>) getAnyextension
						.invoke(eObject);
			} catch (IllegalArgumentException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			} catch (InvocationTargetException e) {
				e.printStackTrace();
			}
		}

		for (Iterator<AnyExtension> iterator = anyextensions.iterator(); iterator
				.hasNext();) {
			AnyExtension object = iterator.next();
			if (object instanceof SAWSDL) {
				result = (SAWSDL) object;
				break;
			}
		}
		return result;
	}

	/**
	 * Get the anyExtensionGroup structural corresponding to the type of a model
	 * element.
	 * 
	 * @param modelElement
	 * @return an AnyExtensionGroup EstrucuralFeature
	 */
	private EStructuralFeature getAnyExtensionGroup(EObject modelElement) {
		EStructuralFeature anyExtensionGroup = null;
		if (modelElement instanceof Component) {
			anyExtensionGroup = ScaPackage.Literals.COMPONENT__ANYEXTENSION_GROUP;
		} else if (modelElement instanceof ComponentService) {
			anyExtensionGroup = ScaPackage.Literals.BASE_SERVICE__ANYEXTENSION_GROUP;
		} else if (modelElement instanceof ComponentReference) {
			anyExtensionGroup = ScaPackage.Literals.BASE_REFERENCE__ANYEXTENSION_GROUP;
		}
		return anyExtensionGroup;
	}

	/**
	 * Get the model object corresponding to an EditPart.
	 * 
	 * @param editPart
	 * @return the current model object
	 */
	private EObject getCurrentModelObject(GraphicalEditPart editPart) {
		EObject current = null;
		if (editPart.getModel() instanceof View) {
			current = ((View) editPart.getModel()).getElement();
		}
		return current;
	}

	/**
	 * Extract the modelReference from a request.
	 * 
	 * @param request
	 * @return the modelReference
	 */
	private String extractModelReference(Request request) {
		if (request != null
				&& ((DropObjectsRequest) request).getObjects().size() == 1
				&& ((DropObjectsRequest) request).getObjects().get(0) instanceof String) {
			String modelReference = ((DropObjectsRequest) request).getObjects()
					.get(0).toString();
			if (modelReference.startsWith("modelReference=")) { //$NON-NLS-1$
				return modelReference.substring("modelReference=\"".length(), //$NON-NLS-1$
						modelReference.length() - 1);
			} else
				return null;
		}
		return null;
	}

	/**
	 * Check if this IDragDropHandler is enabled
	 * 
	 * @param event
	 * @param part
	 * @return true if this IDragDropHandler is enabled.
	 */
	public boolean isEnabled(DropTargetEvent event, EditPart part) {
		if (part != null && part.getModel() instanceof View) {
			EObject modelElement = ((View) part.getModel()).getElement();
			return isSupportedType(modelElement) && isSupportedEvent(event);
		}
		return false;
	}

	/**
	 * Check if the type of a model element is supported.
	 * 
	 * @param modelElement
	 * @return true if the type of a modelElement is supported.
	 */
	private boolean isSupportedType(EObject modelElement) {
		return (modelElement instanceof Component
				|| modelElement instanceof ComponentService || modelElement instanceof ComponentReference);
	}

	/**
	 * Check if an event is supported.
	 * 
	 * @param event
	 * @return if event is supported
	 */
	private boolean isSupportedEvent(DropTargetEvent event) {
		return event.data == null
				|| ((String) event.data).startsWith("modelReference="); //$NON-NLS-1$
	}
}