/*******************************************************************************
 * Copyright (c) 2004,2009 Actuate Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Actuate Corporation  - initial API and implementation
 *******************************************************************************/

package org.eclipse.birt.core.archive.compound;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.birt.core.archive.ArchiveUtil;
import org.eclipse.birt.core.archive.IDocArchiveWriter;
import org.eclipse.birt.core.archive.IStreamSorter;
import org.eclipse.birt.core.archive.RAInputStream;
import org.eclipse.birt.core.archive.RAOutputStream;

public class ArchiveWriter implements IDocArchiveWriter
{

	boolean shareArchive;
	IArchiveFile archive;

	public ArchiveWriter( String archiveName ) throws IOException
	{
		archive = new ArchiveFile( archiveName, "rw" );
		shareArchive = false;
	}

	public ArchiveWriter( IArchiveFile archive ) throws IOException
	{
		this.archive = archive;
		shareArchive = true;
	}

	public IArchiveFile getArchive( )
	{
		return archive;
	}

	public RAOutputStream createRandomAccessStream( String relativePath )
			throws IOException
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		ArchiveEntry entry = archive.createEntry( relativePath );
		return new ArchiveEntryOutputStream( entry );
	}

	public RAOutputStream openRandomAccessStream( String relativePath )
			throws IOException
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		ArchiveEntry entry;
		if ( archive.exists( relativePath ) )
		{
			entry = archive.openEntry( relativePath );
		}
		else
		{
			entry = archive.createEntry( relativePath );
		}
		return new ArchiveEntryOutputStream( entry );
	}

	public RAOutputStream createOutputStream( String relativePath )
			throws IOException
	{
		return createRandomAccessStream( relativePath );
	}

	public RAOutputStream getOutputStream( String relativePath )
			throws IOException
	{
		return openRandomAccessStream( relativePath );
	}

	public RAInputStream getInputStream( String relativePath )
			throws IOException
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		ArchiveEntry entry = archive.openEntry( relativePath );
		return new ArchiveEntryInputStream( entry );
	}

	public boolean dropStream( String relativePath )
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		try
		{
			return archive.removeEntry( relativePath );
		}
		catch ( IOException ex )
		{
			return false;
		}
	}

	public boolean exists( String relativePath )
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		return archive.exists( relativePath );
	}

	public void finish( ) throws IOException
	{
		try
		{
			// flush the archvies
			archive.flush( );
		}
		finally
		{
			if ( !shareArchive )
			{
				archive.close( );
			}
		}
	}

	public void flush( ) throws IOException
	{
		archive.flush( );
	}

	public String getName( )
	{
		return archive.getName( );
	}

	public void initialize( ) throws IOException
	{
	}

	public void setStreamSorter( IStreamSorter streamSorter )
	{
	}

	public Object lock( String relativePath ) throws IOException
	{
		if ( !relativePath.startsWith( ArchiveUtil.UNIX_SEPERATOR ) )
			relativePath = ArchiveUtil.UNIX_SEPERATOR + relativePath;
		return archive.lockEntry( relativePath );
	}

	public void unlock( Object locker )
	{
		try
		{
			archive.unlockEntry( locker );
		}
		catch ( IOException ex )
		{
		}
	}

	public List listAllStreams( ) throws IOException
	{
		ArrayList<String> list = new ArrayList<String>( );
		list.addAll( archive.listEntries( "/" ) );
		return list;
	}

	public List listStreams( String namePattern ) throws IOException
	{
		ArrayList list = new ArrayList( );
		Iterator iter = archive.listEntries( "/" ).iterator( );
		while ( iter.hasNext( ) )
		{
			String name = (String) iter.next( );
			if ( name.startsWith( namePattern )
					&& !name.equalsIgnoreCase( namePattern ) )
			{
				String diffString = ArchiveUtil.generateRelativePath(
						namePattern, name );
				if ( diffString.lastIndexOf( ArchiveUtil.UNIX_SEPERATOR ) == 0 )
				{
					list.add( name );
				}
			}
		}
		return list;
	}
}
