/*******************************************************************************
 * Copyright (c) 2009, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.emf.facet.infra.facet.ui.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;

import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.infra.facet.FacetPackage;
import org.eclipse.emf.facet.infra.facet.FacetSet;
import org.eclipse.emf.facet.infra.facet.core.FacetSetCatalog;
import org.eclipse.emf.facet.infra.facet.ui.Activator;
import org.eclipse.emf.facet.infra.facet.ui.dialogs.LoadFacetsDialog;
import org.eclipse.emf.facet.infra.facet.ui.views.FacetSetContentProvider;
import org.eclipse.emf.facet.util.emf.ui.internal.utils.ImageUtils;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.graphics.Image;

public abstract class AbstractLoadFacetsAction extends Action {

	private static final String ICON_FACET = "icons/Facet.gif"; //$NON-NLS-1$

	public AbstractLoadFacetsAction() {
		super("Load Facets", IAction.AS_PUSH_BUTTON);
		setToolTipText("Load Facets");
		setImageDescriptor(Activator.createImageDescriptor(AbstractLoadFacetsAction.ICON_FACET));
	}

	@Override
	public void run() {
		BusyIndicator.showWhile(null, new Runnable() {
			public void run() {
				FacetSetCatalog.getSingleton().waitUntilBuilt();
			}
		});
		openDialog();
	}

	protected void openDialog() {
		final ITreeContentProvider treeContentProvider = new FacetSetContentProvider(
				FacetPackage.eINSTANCE.getFacet());

		final LoadFacetsDialog dialog = new LoadFacetsDialog(null, this.labelProvider,
				treeContentProvider, getReferencedEPackages());
		dialog.setContainerMode(true);
		dialog.setHelpAvailable(false);
		dialog.setTitle("Load Facets");
		dialog.setMessage("Choose Facets to Load");

		final Collection<FacetSet> facetSets = FacetSetCatalog.getSingleton().getAllFacetSets();
		dialog.setInput(facetSets.toArray());
		dialog.setInitialElementSelections(new ArrayList<Facet>(getLoadedFacets()));

		if (dialog.open() == Window.OK) {
			final Object[] result = dialog.getResult();
			clearFacets();
			final HashSet<Facet> facetsToLoad = new HashSet<Facet>();
			for (final Object resultElement : result) {
				if (resultElement instanceof Facet) {
					final Facet facet = (Facet) resultElement;
					facetsToLoad.add(facet);
				} else if (!(resultElement instanceof FacetSet)) {
					throw new RuntimeException("Unexpected element type"); //$NON-NLS-1$
				}
			}

			BusyIndicator.showWhile(null, new Runnable() {
				public void run() {
					for (Facet facetToLoad : facetsToLoad) {
						loadFacet(facetToLoad);
					}

					facetsLoaded(facetsToLoad);
				}

			});
		}
	}

	/** @param facets */
	protected void facetsLoaded(final Collection<Facet> facets) {
		//
	}

	protected abstract void loadFacet(final Facet facetToLoad);

	protected abstract void clearFacets();

	protected abstract Collection<? extends Facet> getLoadedFacets();

	protected abstract Collection<EPackage> getReferencedEPackages();

	private final LabelProvider labelProvider = new LabelProvider() {
		@Override
		public String getText(final Object element) {
			String label;
			if (element instanceof FacetSet) {
				final FacetSet facetSet = (FacetSet) element;
				label = facetSet.getName();
			} else if (element instanceof Facet) {
				final Facet facet = (Facet) element;
				label = facet.getName();
			} else if (element == null) {
				label = "null"; //$NON-NLS-1$
			} else {
				label = element.getClass().getName();
			}
			return label;
		}

		@Override
		public Image getImage(final Object element) {
			return ImageUtils.getImage(element);
		}
	};
}
