/*******************************************************************************
 * Copyright (c) 2010, 2011 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *   Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *   Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *   Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *   Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *   Nicolas Guyomar (Mia-Software) - Bug 339554 - org.eclipse.emf.facet.widgets.celleditors API cleaning
 *   Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *   Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import net.sourceforge.nattable.data.IDataProvider;
import net.sourceforge.nattable.sort.SortDirectionEnum;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.infra.facet.FacetStructuralFeature;
import org.eclipse.emf.facet.infra.facet.core.FacetContext;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FeatureColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Row;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.QueryRow;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Shell;

class BodyDataProvider implements IDataProvider {
	private final int columnCount;
	private final List<Row> originalElements;
	private List<Row> elements;
	private final List<Column> columns;
	private final Shell shell;
	private final EditingDomain editingDomain;
	private final TableLabelProvider tableLabelProvider;
	private final FacetContext facetContext;

	public BodyDataProvider(final EList<Row> rowsList, final int columnCount,
			final List<Column> columns, final Shell shell,
			final EditingDomain editingDomain, final TableLabelProvider labelProvider, final FacetContext facetContext) {
		this.columns = columns;
		this.shell = shell;
		this.editingDomain = editingDomain;
		this.tableLabelProvider = labelProvider;
		this.originalElements = rowsList;
		this.elements = new ArrayList<Row>(rowsList);
		this.columnCount = columnCount;
		this.facetContext = facetContext;
	}

	public Row getRow(final int row) {
		return this.elements.get(row);
	}

	public int getRowCount() {
		return this.elements.size();
	}

	public int getColumnCount() {
		return this.columnCount;
	}

	public Object getDataValue(final int columnIndex, final int rowIndex) {
		Column column = this.columns.get(columnIndex);
		Row row = this.elements.get(rowIndex);
		if (row instanceof QueryRow) {
			QueryRow tableElementWithContext = (QueryRow) row;
			return new GridElementWithContext(row.getElement(),
					tableElementWithContext.getQueryResult().getSource(), column);
		}
		return new GridElement(row.getElement(), column);
	}

	public void setDataValue(final int columnIndex, final int rowIndex, final Object newValue) {
		Column column = this.columns.get(columnIndex);
		Object element = this.elements.get(rowIndex).getElement();
		if (element instanceof EObject) {
			EObject eObject = (EObject) element;

			EStructuralFeature feature;
			if (column instanceof FeatureColumn) {
				feature = ((FeatureColumn) column).getFeature();
			} else {
				throw new RuntimeException("Cannot edit in column of type " + column.eClass().getName()); //$NON-NLS-1$
			}

			if (!feature.isChangeable()) {
				throw new RuntimeException("Feature is not changeable: " + feature.getName()); //$NON-NLS-1$
			}

			Object prevValue = null;
			if (feature instanceof FacetStructuralFeature) {
				try {
					prevValue = this.facetContext.get(eObject, feature);
				}  catch (Exception e) {
					throw new RuntimeException(e);
				}
			} else {
				prevValue = eObject.eGet(feature);
			}
			if (prevValue == newValue) {
				return;
			}

			if (newValue == null && !feature.isUnsettable()) {
				MessageDialog.openWarning(
						this.shell,
						Messages.BodyDataProvider_cannotUnset,
						NLS.bind(Messages.BodyDataProvider_featureIsNotUnsettable,
								feature.getName()));
				return;
			}

			if (feature instanceof EReference && ((EReference) feature).isContainment()) {
				Object object = eObject.eGet(feature);
				boolean empty = (object == null || object instanceof List
						&& ((List<?>) object).isEmpty());
				String message;
				if (empty) {
					message = Messages.BodyDataProvider_valueWillBeMoved;
				} else {
					message = Messages.BodyDataProvider_valueWillBeMovedAndPreviousLost;
				}
				String fullMessage = Messages.BodyDataProvider_featureIsComposite + message;

				if (!MessageDialog.openConfirm(this.shell, Messages.BodyDataProvider_moveElement,
						fullMessage)) {
					return;
				}
			}

			if (feature instanceof FacetStructuralFeature) {
				try {
					this.facetContext.set(eObject, (FacetStructuralFeature) feature, newValue,
							this.editingDomain);
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			} else {
				ICommandFactory commandFactory = ICommandFactoriesRegistry.INSTANCE
						.getCommandFactoryFor(this.editingDomain);
				Command setCommand = commandFactory.createSetCommand(this.editingDomain, eObject,
						feature, newValue);
				this.editingDomain.getCommandStack().execute(setCommand);
			}
		}

		// not editable
	}

	public void sort(final int columnIndex, final SortDirectionEnum sortDirection) {
		if (sortDirection == SortDirectionEnum.NONE) {
			this.elements = new ArrayList<Row>(this.originalElements);
		} else {
			Collections.sort(this.elements, new Comparator<Row>() {
				public int compare(final Row t1, final Row t2) {
					Column columnDescription = BodyDataProvider.this.columns
							.get(columnIndex);
					String label1 = BodyDataProvider.this.tableLabelProvider.getColumnText(
							new GridElement(t1.getElement(), columnDescription), columnIndex);
					String label2 = BodyDataProvider.this.tableLabelProvider.getColumnText(
							new GridElement(t2.getElement(), columnDescription), columnIndex);
					if (sortDirection == SortDirectionEnum.ASC) {
						return label1.compareToIgnoreCase(label2);
					}
					return label2.compareToIgnoreCase(label1);
				}
			});
		}
	}
}