/*******************************************************************************
 * Copyright (c) 2011 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import java.util.ArrayList;
import java.util.List;

import net.sourceforge.nattable.data.convert.IDisplayConverter;
import net.sourceforge.nattable.data.validate.IDataValidator;
import net.sourceforge.nattable.edit.ICellEditHandler;
import net.sourceforge.nattable.edit.editor.ICellEditor;
import net.sourceforge.nattable.style.IStyle;

import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.infra.facet.FacetStructuralFeature;
import org.eclipse.emf.facet.infra.facet.core.FacetContext;
import org.eclipse.emf.facet.infra.facet.core.exception.EmfFacetFacetException;
import org.eclipse.emf.facet.infra.query.core.exception.ModelQueryException;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactoriesRegistry;
import org.eclipse.emf.facet.widgets.celleditors.ICommandFactory;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditHandler;
import org.eclipse.emf.facet.widgets.celleditors.IModelCellEditor;
import org.eclipse.emf.facet.widgets.celleditors.INaryEReferenceCellEditor;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FeatureColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

/** Adapts an {@link IModelCellEditor} to an {@link ICellEditor} */
public class NatTableNaryEReferenceCellEditorAdapter<T extends EObject> implements ICellEditor {

	private final INaryEReferenceCellEditor<T> cellEditor;
	private final List<Column> columns;
	private Control control;
	private Composite composite;
	private FacetContext facetContext;
	private EditingDomain editingDomain;

	public NatTableNaryEReferenceCellEditorAdapter(final INaryEReferenceCellEditor<T> cellEditor,
			final List<Column> columns, final FacetContext facetContext,
			final EditingDomain editingDomain) {
		this.cellEditor = cellEditor;
		this.columns = columns;
		this.facetContext = facetContext;
		this.editingDomain = editingDomain;
	}

	public Control activateCell(final Composite parent, final Object originalCanonicalValue,
			final Character initialEditValue, final IDisplayConverter displayConverter,
			final IStyle cellStyle, final IDataValidator dataValidator,
			final ICellEditHandler editHandler, final int colIndex, final int rowIndex) {

		this.composite = parent;

		Column column = this.columns.get(colIndex);
		if (!(column instanceof FeatureColumn)) {
			// error (when called for multi-cell editing)
			this.control = new Composite(parent, SWT.NONE);
			return this.control;
		}
		FeatureColumn featureColumn = (FeatureColumn) column;
		final EStructuralFeature feature = featureColumn.getFeature();

		GridElement gridElement = (GridElement) originalCanonicalValue;
		Object element = gridElement.getElement();
		if (element instanceof EObject) {
			final EObject eObject = (EObject) element;
			try {
				if (feature instanceof FacetStructuralFeature) {
					IModelCellEditHandler modelCellEditHandler = new IModelCellEditHandler() {
						public void commit() {
							List<T> value = NatTableNaryEReferenceCellEditorAdapter.this.cellEditor
									.getValue();
							try {
								NatTableNaryEReferenceCellEditorAdapter.this.facetContext.set(
										eObject, (FacetStructuralFeature) feature, value,
										NatTableNaryEReferenceCellEditorAdapter.this.editingDomain);
							} catch (Exception e) {
								throw new RuntimeException(e);
							}
						}
					};
					List<T> values = this.listCastedFacetGet(eObject, feature);
					this.control = this.cellEditor.activateCell(parent, values,
							getAvailableValues(eObject, feature), modelCellEditHandler,
							eObject, feature);

				} else {
					IModelCellEditHandler modelCellEditHandler = new IModelCellEditHandler() {
						public void commit() {
							List<T> value = NatTableNaryEReferenceCellEditorAdapter.this.cellEditor
									.getValue();
							ICommandFactory commandFactory = ICommandFactoriesRegistry.INSTANCE
									.getCommandFactoryFor(NatTableNaryEReferenceCellEditorAdapter.this.editingDomain);

							Command setCommand = commandFactory.createSetCommand(
									NatTableNaryEReferenceCellEditorAdapter.this.editingDomain,
									eObject, feature, value);
							NatTableNaryEReferenceCellEditorAdapter.this.editingDomain
									.getCommandStack().execute(setCommand);
						}
					};
					@SuppressWarnings("unchecked")
					//unchecked: We can not check that the eGet returns a List of T.
					List<T> values = (List<T>) eObject.eGet(feature);
					this.control = this.cellEditor.activateCell(parent, values,
							getAvailableValues(eObject, feature), modelCellEditHandler, eObject,
							feature);
				}
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
			return this.control;
		}
		throw new IllegalArgumentException("not an EObject"); //$NON-NLS-1$
	}

	@SuppressWarnings("unchecked")
	//unchecked: We can not check that the method FacetContext.get(EObject, EStructuralFeature) returns a List of T.
	private List<T> listCastedFacetGet(final EObject eObject, final EStructuralFeature feature)
			throws ModelQueryException, EmfFacetFacetException {
		return (List<T>) this.facetContext.get(eObject, feature);
	}
	
	private List<T> getAvailableValues(final EObject source, final EStructuralFeature feature)
			throws ModelQueryException, EmfFacetFacetException {
		EClassifier eType = feature.getEType();
		List<?> featureValues = null;
		if (feature instanceof FacetStructuralFeature) {
			featureValues = this.listCastedFacetGet(source, feature);
		} else {
			featureValues = (List<?>) source.eGet(feature);
		}
		// build a list of EObjects assignable to the EReference
		List<T> list = new ArrayList<T>();
		Resource eResource = source.eResource();
		if (eResource == null) {
			return new ArrayList<T>();
		}
		ResourceSet resourceSet = eResource.getResourceSet();
		TreeIterator<Notifier> allContents = resourceSet.getAllContents();
		while (allContents.hasNext()) {
			Notifier notifier = allContents.next();
			if (notifier instanceof EObject) {
				EObject eObject = (EObject) notifier;
				if (eType.isInstance(eObject)
						&& (!feature.isUnique() || !(featureValues != null && featureValues
								.contains(eObject)))) {
						@SuppressWarnings("unchecked")
						//unchecked: This cast can not be cleanly check using Java but it is done by the test eType.isInstance(eObject)
						T t = (T) eObject; 
						list.add(t);
				}
			}
		}
		return list;
	}

	public void setCanonicalValue(final Object canonicalValue) {
		// no reason to set the value after creation
		throw new UnsupportedOperationException();
	}

	public Object getCanonicalValue() {
		// should never be called
		throw new UnsupportedOperationException(
				"commit should be disabled and editing domain used directly"); //$NON-NLS-1$
	}

	public void close() {
		if (this.composite != null && !this.composite.isDisposed()) {
			this.composite.forceFocus();
		}
		if (this.control != null && !this.control.isDisposed()) {
			this.control.dispose();
		}
	}

	public boolean isClosed() {
		return this.control == null || this.control.isDisposed();
	}

}
