/*******************************************************************************
 * Copyright (c) 2009, 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 331900 - customizable NatTable
 *    Nicolas Bros (Mia-Software) - Bug 332215 - customizable NatTable column headers
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - To be able to save the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Gregoire Dupe (Mia-Software) - Bug 332998 - To be able to add a column and fill it with the result of a query
 *    Nicolas Guyomar (Mia-Software) - Bug 337398 - [Add a Query Column] Cannot modify resource set without a write transaction
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation 
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Nicolas Guyomar (Mia-Software) - Bug 342451 - To be able to edit derived facet attributes and derived facet references in a table
 *    Nicolas Guyomar (Mia-Software) - Bug 344670 - Problems with the columns creation : very slow + scrollbar blinked
 *    Nicolas Guyomar (Mia-Software) - Bug 346733 - Bug in TableLabelProvider : when the column is a FacetColumn the returned label is not correct
 *    Gregoire Dupe (Mia-Software) - Bug 350700 - TableWidget: Wrong cell label in multi-valued facet reference columns
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.internal;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.facet.infra.browser.uicore.internal.customization.CustomizationEngine;
import org.eclipse.emf.facet.infra.common.core.internal.utils.StringUtils;
import org.eclipse.emf.facet.infra.facet.Facet;
import org.eclipse.emf.facet.infra.facet.FacetAttribute;
import org.eclipse.emf.facet.infra.facet.FacetReference;
import org.eclipse.emf.facet.infra.facet.core.FacetContext;
import org.eclipse.emf.facet.infra.query.core.AbstractModelQuery;
import org.eclipse.emf.facet.infra.query.core.ModelQuerySetCatalog;
import org.eclipse.emf.facet.infra.query.core.exception.ModelQueryException;
import org.eclipse.emf.facet.infra.query.runtime.ModelQueryResult;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.internal.CustomizableLabelProvider;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.AttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.Column;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ContextColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.DefaultLabelColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.EContainerColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetAttributeColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.FacetReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.MetaClassColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.QueryColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.ReferenceColumn;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance2.ValueColumn;
import org.eclipse.swt.graphics.Image;

public class TableLabelProvider {

	private static final int MAX_DISPLAYED_ELEMENTS = 5;
	private final List<Column> columns;
	private final CustomizableLabelProvider labelProvider;
	private FacetContext facetContext;

	public TableLabelProvider(final List<Column> columns,
			final CustomizationEngine customizationEngine, final FacetContext facetContext) {
		this.columns = columns;
		this.labelProvider = new CustomizableLabelProvider(customizationEngine);
		this.facetContext = facetContext;
	}

	public String getTextFor(final Object element) {
		if (element == null) {
			return "null"; //$NON-NLS-1$
		}
		if (element instanceof EObject) {
			final EObject eObject = (EObject) element;
			return this.labelProvider.getText(eObject);
		}
		// primitive DataType
		return element.toString();
	}

	public Image getImageFor(final Object element) {
		if (element == null) {
			return null;
		}

		if (element instanceof EObject) {
			final EObject eObject = (EObject) element;
			return this.labelProvider.getImage(eObject);
		}

		return null;
	}

	public Image getColumnImage(final Object element, final int columnIndex) {
		if (columnIndex < this.columns.size()) {
			final Column columnDescription = this.columns.get(columnIndex);
			GridElement gridElement = (GridElement) element;

			Object gridElementElement = gridElement.getElement();
			if (gridElementElement == null) {
				return null;
			}
			if (columnDescription instanceof DefaultLabelColumn) {
				return getImageFor(gridElement.getElement());
			} else if (columnDescription instanceof ReferenceColumn) {
				final EReference reference = ((ReferenceColumn) columnDescription).getReference();
				final EObject eObject = (EObject) gridElement.getElement();

				final Object refValue;
				if (!eObject.eClass().getEAllReferences().contains(reference)) {
					return null;
				}
				refValue = eObject.eGet(reference);
				// }

				if (reference.isMany()) {
					@SuppressWarnings("unchecked")
					final EList<EObject> list = (EList<EObject>) refValue;

					// when only 1 element, then display its icon directly
					if (list.size() == 1) {
						return getImageFor(list.get(0));
					}
					return null;
				}
				if (refValue != null) {
					return getImageFor(refValue);
				}
			} else if (columnDescription instanceof EContainerColumn) {
				return getImageFor(((EObject) gridElement.getElement()).eContainer());
			} else if (columnDescription instanceof ContextColumn) {
				return getContextColumnImage(gridElement);
			} else {
				return null;
			}
		}

		return null;
	}

	public String getColumnText(final GridElement gridElement, final int columnIndex) {
		if (columnIndex >= this.columns.size()) {
			throw new IndexOutOfBoundsException();
		}
		final Column column = this.columns.get(columnIndex);

		Object element = gridElement.getElement();
		if (element == null) {
			return "null"; //$NON-NLS-1$
		}
		if (column instanceof DefaultLabelColumn) {
			return getTextFor(gridElement.getElement());
		} else if (column instanceof ReferenceColumn) {
			return getReferenceColumnText(gridElement.getElement(), column);
		} else if (column instanceof EContainerColumn) {
			return getTextFor(((EObject) gridElement.getElement()).eContainer());
		} else if (column instanceof ContextColumn) {
			return getContextColumnText(gridElement);
		} else if (column instanceof AttributeColumn) {
			return getAttributeColumnText(gridElement.getElement(), column);
		} else if (column instanceof MetaClassColumn) {
			return getMetaclassColumnText(gridElement.getElement());
		} else if (column instanceof QueryColumn) {
			if (gridElement.getElement() instanceof EObject) {
				final EObject eObject = (EObject) gridElement.getElement();
				QueryColumn queryColumn = (QueryColumn) column;
				ModelQueryResult queryResult = null;
				try {
					final AbstractModelQuery abstractModelQuery = ModelQuerySetCatalog
							.getSingleton().getModelQueryImpl(queryColumn.getQuery());
					queryResult = abstractModelQuery.evaluate(eObject);
				} catch (ModelQueryException e) {
					Logger.logWarning(e, Activator.getDefault());
					return "N/A"; //$NON-NLS-1$
				}
				if (queryResult.getException() != null) {
					return "N/A"; //$NON-NLS-1$
				}
				if (queryResult.getValue() == null) {
					return "null"; //$NON-NLS-1$
				}
				return queryResultToString(queryResult.getValue());
			}
			return "N/A"; //$NON-NLS-1$

		} else if (column instanceof FacetAttributeColumn) {
			FacetAttributeColumn facetAttributeColumn = (FacetAttributeColumn) column;
			try {
				EObject eObject = (EObject) gridElement.getElement();
				FacetAttribute facetAttribute = facetAttributeColumn.getAttribute();
				if (this.facetContext.getAppliedFeatures().contains(facetAttribute)) {
					Facet facet = (Facet) facetAttribute.eContainer();
					if (this.facetContext.isInstance(eObject, facet)) {
						return queryResultToString(this.facetContext.get(eObject, facetAttribute));
					}
				}
				return Messages.TableLabelProvider_NA;
			} catch (Exception e) {
				Logger.logError(e, Activator.getDefault());
				return '!' + e.getClass().getName() + "(cf. log)"; //$NON-NLS-1$
			}

		} else if (column instanceof FacetReferenceColumn) {
			FacetReferenceColumn facetReferenceColumn = (FacetReferenceColumn) column;
			try {
				EObject eObject = (EObject) gridElement.getElement();
				FacetReference facetReference = facetReferenceColumn.getReference();
				if (this.facetContext.getAppliedFeatures().contains(facetReference)) {
					Facet facet = (Facet) facetReference.eContainer();
					if (this.facetContext.isInstance(eObject, facet)) {
						return queryResultToString(this.facetContext.get(eObject, facetReference));
					}
				}
				return Messages.TableLabelProvider_NA;

			} catch (Exception e) {
				Logger.logError(e, Activator.getDefault());
				return '!' + e.getClass().getName() + "(cf. log)"; //$NON-NLS-1$
			}
		} else if (column instanceof ValueColumn) {
			return queryResultToString(gridElement.getElement());
		} else {
			throw new IllegalArgumentException();
		}
	}
	
	private String queryResultToString(final Object object) {
		if (object == null) {
			return "null"; //$NON-NLS-1$
		}
		if (object instanceof Collection<?>) {
			StringBuilder stb = new StringBuilder();
			Collection<?> collection = (Collection<?>) object;
			for (Object o : collection) {
				if (stb.length() > 0) {
					stb.append(", "); //$NON-NLS-1$
				} 
				if (o instanceof EObject) {
					stb.append(getTextFor(o));
				} else {
					stb.append(o.toString());
				}
			}
			return stb.toString();
		} else if (object instanceof ModelQueryResult) {
			return getTextFor(((ModelQueryResult) object).getValue());
		}
		return getTextFor(object);
	}

	private static String getMetaclassColumnText(final Object element) {
		if (element == null) {
			return ""; //$NON-NLS-1$
		}
		if (element instanceof EObject) {
			EObject eObject = (EObject) element;
			final EClass eClass = eObject.eClass();
			if (eClass == null) {
				return Messages.TableLabelProvider_none;
			}
			return eClass.getName();
		}
		// primitive DataType
		return element.getClass().getSimpleName();
	}

	private String getContextColumnText(final GridElement gridElement) {
		if (gridElement instanceof GridElementWithContext) {
			GridElementWithContext gridElementWithContext = (GridElementWithContext) gridElement;
			EObject context = gridElementWithContext.getContext();
			return getTextFor(context);
		}
		return Messages.TableLabelProvider_null;
	}

	private Image getContextColumnImage(final GridElement gridElement) {
		if (gridElement instanceof GridElementWithContext) {
			GridElementWithContext gridElementWithContext = (GridElementWithContext) gridElement;
			EObject context = gridElementWithContext.getContext();
			return getImageFor(context);
		}
		return null;
	}

	private String getAttributeColumnText(final Object element, final Column column) {
		final EObject eObject = (EObject) element;
		EAttribute attribute = null;
		if (column instanceof AttributeColumn) {
			AttributeColumn attributeColumn = (AttributeColumn) column;
			attribute = attributeColumn.getAttribute();
		}
		final Object attrValue;
		if (!eObject.eClass().getEAllAttributes().contains(attribute)) {
			return Messages.TableLabelProvider_NA;
		}
		attrValue = eObject.eGet(attribute);
		// }

		if (attrValue == null) {
			return "null"; //$NON-NLS-1$
		}
		return StringUtils.truncateBeforeNewline(getTextFor(attrValue));
	}

	private String getReferenceColumnText(final Object element, final Column column) {
		final EObject eObject = (EObject) element;
		EReference reference = null;
		if (column instanceof ReferenceColumn) {
			ReferenceColumn referenceColumn = (ReferenceColumn) column;
			reference = referenceColumn.getReference();
		}
		if (reference != null) {
			final Object refValue;

			if (!eObject.eClass().getEAllReferences().contains(reference)) {
				return Messages.TableLabelProvider_NA;
			}
			refValue = eObject.eGet(reference);

			if (refValue == null) {
				return "null"; //$NON-NLS-1$
			} else if (reference.isMany()) {
				@SuppressWarnings("unchecked")
				final EList<EObject> list = (EList<EObject>) refValue;
				if (list.size() == 0) {
					return ""; //$NON-NLS-1$
				}
				// when only 1 element, then display it directly
				if (list.size() == 1) {
					return getTextFor(list.get(0));
				}
				StringBuilder builder = new StringBuilder();
				for (int i = 0; i < list.size() && i < TableLabelProvider.MAX_DISPLAYED_ELEMENTS; i++) {
					if (builder.length() > 0) {
						builder.append(", "); //$NON-NLS-1$
					}
					builder.append(getTextFor(list.get(i)));
				}
				if (list.size() > TableLabelProvider.MAX_DISPLAYED_ELEMENTS) {
					builder.append(", ..."); //$NON-NLS-1$
				}
				return builder.toString();
			} else {
				return getTextFor(refValue);
			}
		}
		return "null"; //$NON-NLS-1$
	}
}
