/*******************************************************************************
 * Copyright (c) 2010, 2011 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - Bug 331203 - table model editor - initial API and implementation
 *    Nicolas Bros (Mia-Software) - Bug 332437 - NatTable : pluggable cell editors
 *    Nicolas Bros (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Guyomar (Mia-Software) - Bug 332226 - To be able to create or delete model element from the table
 *    Nicolas Bros (Mia-Software) - Bug 332438 - NatTable : table type
 *    Nicolas Guyomar (Mia-Software) - Bug 332924 - The user must be able to save which EMF objects are contained in a table
 *    Nicolas Guyomar (Mia-Software) - Bug 333029 - To be able to save the size of the lines and the columns
 *    Nicolas Guyomar (Mia-Software) - Bug 333414 - The user must be able to save the column order
 *    Nicolas Guyomar (Mia-Software) - Bug 335020 - Nattable widget should use the Eclipse framework
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation 
 *    Nicolas Guyomar (Mia-Software) - Bug 340738 - Utility method to create a coherent tableInstance
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.nattable.workbench.internal.editor;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.facet.infra.query.ModelQuery;
import org.eclipse.emf.facet.infra.query.runtime.ModelQueryResult;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.nattable.INatTableUtils;
import org.eclipse.emf.facet.widgets.nattable.NatTableWidgetUtils;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.QueryTableInstance;
import org.eclipse.emf.facet.widgets.nattable.instance.tableinstance.TableInstance;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.TableConfiguration;
import org.eclipse.emf.facet.widgets.nattable.tableconfiguration.UnsatisfiedConfigurationConstraint;
import org.eclipse.emf.facet.widgets.nattable.workbench.internal.Activator;
import org.eclipse.emf.facet.widgets.nattable.workbench.internal.Messages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

public class NatTableEditorFactory {

	private static NatTableEditorFactory instance;

	protected NatTableEditorFactory() {
		// singleton
	}

	public static NatTableEditorFactory getInstance() {
		if (NatTableEditorFactory.instance == null) {
			NatTableEditorFactory.instance = new NatTableEditorFactory();
		}
		return NatTableEditorFactory.instance;
	}

	/**
	 * Open a new table view editor on the given list of elements.
	 * 
	 * @param browserConfiguration
	 *            the main editor configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table
	 *            view editor
	 * @throws UnsatisfiedConfigurationConstraint
	 */
	public void openOn(final List<EObject> elements, final EditingDomain editingDomain,
			final String description, final TableConfiguration tableConfiguration,
			final EObject context, final Object parameter)
			throws UnsatisfiedConfigurationConstraint {
		TableInstance tableInstance = NatTableWidgetUtils.createTableInstance(elements,
				description, tableConfiguration, context, parameter);
		openOn(tableInstance, editingDomain);
	}

	public void openOn(final TableInstance input, final EditingDomain editingDomain)
			throws UnsatisfiedConfigurationConstraint {
		if (input instanceof QueryTableInstance) {
			this.openOnQueryResult(null, (QueryTableInstance) input);
		} else {
			final TableEditorInput tableEditorInput = new TableEditorInput(input, editingDomain);
			openOn(tableEditorInput);
		}
	}

	private void openOn(final TableEditorInput input) throws UnsatisfiedConfigurationConstraint {
		if (!INatTableUtils.INSTANCE.checkInput(input.getTableInstance())) {
			throw new UnsatisfiedConfigurationConstraint();
		}

		final IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getActivePage();
		try {
			IDE.openEditor(activePage, input, getEditorID(), true);
		} catch (final PartInitException e) {
			Logger.logError(e, Activator.getDefault());
		}
	}

	protected String getEditorID() {
		return NatTableEditor.ID;
	}

	/**
	 * Open a new table view editor on the given query result.
	 * 
	 * @param result
	 *            the results of a query to display in the table
	 * @param modelQuery
	 *            the query that was evaluated
	 * @throws UnsatisfiedConfigurationConstraint
	 */
	public void openOnQueryResult(final List<ModelQueryResult> result, final ModelQuery modelQuery)
			throws UnsatisfiedConfigurationConstraint {
		QueryTableInstance queryTableInstance = NatTableWidgetUtils.createQueryTableInstance(
				result, null, null, null);
		openOnQueryResult(modelQuery, queryTableInstance);
	}

	private void openOnQueryResult(final ModelQuery modelQuery,
			final QueryTableInstance queryTableInstance) throws UnsatisfiedConfigurationConstraint {

		final String description;
		final String queryName;
		int nElements = 0;
		for (ModelQueryResult mqr : queryTableInstance.getQueryResults()) {
			if (mqr.getValue() instanceof Collection<?>) {
				Collection<?> collection = (Collection<?>) mqr.getValue();
				nElements += collection.size();
			}
		}

		if (modelQuery != null) {
			queryName = modelQuery.getName();
			description = NLS.bind(Messages.NatTableEditorFactory_resultsForQuery, queryName,
					new Integer(nElements));
		} else {
			if (nElements > 1) {
				description = NLS.bind(Messages.NatTableEditorFactory_queryResults, new Integer(
						nElements));
			} else {
				description = NLS.bind(Messages.NatTableEditorFactory_queryResult, new Integer(
						nElements));
			}
			queryName = Messages.NatTableEditorFactory_aQuery;
		}

		queryTableInstance.setDescription(description);

		final TableEditorInput input = new TableEditorInput(queryTableInstance, null) {
			@Override
			public String getName() {

				return queryName;

			}
		};

		openOn(input);
	}
}
