/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.gyrex.cloud.internal.locking;

import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;
import org.eclipse.core.runtime.IPath;
import org.eclipse.gyrex.cloud.internal.CloudDebug;
import org.eclipse.gyrex.cloud.internal.CloudState;
import org.eclipse.gyrex.cloud.internal.NodeInfo;
import org.eclipse.gyrex.cloud.internal.locking.LockAcquirationFailedException;
import org.eclipse.gyrex.cloud.internal.zk.ZooKeeperBasedService;
import org.eclipse.gyrex.cloud.internal.zk.ZooKeeperGate;
import org.eclipse.gyrex.cloud.internal.zk.ZooKeeperMonitor;
import org.eclipse.gyrex.cloud.services.locking.IDistributedLock;
import org.eclipse.gyrex.cloud.services.locking.ILockMonitor;
import org.eclipse.gyrex.common.identifiers.IdHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ZooKeeperLock<T extends IDistributedLock>
extends ZooKeeperBasedService
implements IDistributedLock {
    private static final String SEPARATOR = "__";
    private static final String LOCK_NAME_PREFIX = "lock-";
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperLock.class);
    final ZooKeeperMonitor killMonitor = new ZooKeeperMonitor(){

        @Override
        protected void closing(KeeperException.Code reason) {
            if (!ZooKeeperLock.this.isValid() || ZooKeeperLock.this.isClosed()) {
                return;
            }
            LOG.warn("Lock {} has been lost due to disconnect (reason {})!", (Object)ZooKeeperLock.this.getId(), (Object)reason);
            ZooKeeperLock.this.killLock(KillReason.ZOOKEEPER_DISCONNECT);
        }

        @Override
        protected void pathDeleted(String path) {
            if (!ZooKeeperLock.this.isValid() || ZooKeeperLock.this.isClosed()) {
                return;
            }
            LOG.warn("Lock {} has been deleted on the lock server!", (Object)ZooKeeperLock.this.getId());
            ZooKeeperLock.this.killLock(KillReason.LOCK_DELETED);
        }

        @Override
        protected void recordChanged(String path) {
            if (!ZooKeeperLock.this.isValid() || ZooKeeperLock.this.isClosed()) {
                return;
            }
            LOG.warn("Lock {} has been stolen!", (Object)ZooKeeperLock.this.getId());
            ZooKeeperLock.this.killLock(KillReason.LOCK_STOLEN);
        }
    };
    final String lockId;
    final IPath lockNodePath;
    final String lockNodeContent;
    final boolean ephemeral;
    final boolean recovarable;
    private final ILockMonitor<T> lockMonitor;
    volatile String myLockName;
    volatile String myRecoveryKey;
    volatile String activeLockName;

    public static String createRecoveryKey(String lockName, String nodeContent) {
        return lockName.concat(SEPARATOR).concat(nodeContent);
    }

    public static String[] extractRecoveryKeyDetails(String recoveryKey) {
        String[] keySegments = StringUtils.splitByWholeSeparator((String)recoveryKey, (String)SEPARATOR);
        if (keySegments.length < 2) {
            throw new IllegalArgumentException("invalid recovery key format");
        }
        String lockName = keySegments[0];
        String nodeContent = StringUtils.removeStart((String)recoveryKey, (String)lockName.concat(SEPARATOR));
        if (StringUtils.isBlank((String)lockName) || StringUtils.isBlank((String)nodeContent)) {
            throw new IllegalArgumentException("invalid recovery key format");
        }
        return new String[]{lockName, nodeContent};
    }

    private static int getSequenceNumber(String nodeName) {
        return NumberUtils.toInt((String)StringUtils.removeStart((String)nodeName, (String)LOCK_NAME_PREFIX), (int)-1);
    }

    public ZooKeeperLock(String lockId, ILockMonitor<T> lockMonitor, IPath lockNodeParentPath, boolean ephemeral, boolean recovarable) {
        super(200L, 5);
        if (!IdHelper.isValidId((String)lockId)) {
            throw new IllegalArgumentException("invalid lock id; please see IdHelper#isValidId");
        }
        this.lockId = lockId;
        this.lockNodePath = lockNodeParentPath.append(lockId);
        this.lockMonitor = lockMonitor;
        this.ephemeral = ephemeral;
        this.recovarable = recovarable;
        NodeInfo nodeInfo = CloudState.getNodeInfo();
        if (nodeInfo == null) {
            nodeInfo = new NodeInfo();
        }
        try {
            this.lockNodeContent = String.valueOf(nodeInfo.getNodeId()) + SEPARATOR + nodeInfo.getLocation() + SEPARATOR + DigestUtils.shaHex((byte[])UUID.randomUUID().toString().getBytes("US-ASCII"));
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            throw new IllegalStateException("Please use a JVM that supports UTF-8.");
        }
        try {
            this.asLockType();
        }
        catch (ClassCastException e) {
            throw new ClassCastException(String.format("Cannot cast the lock implementation %s to the generic lock type. Please make sure that the implementation implements the interface. %s", this.getClass().getName(), e.getMessage()));
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected final T acquire(long timeout, boolean recover, String recoveryKey) throws InterruptedException, TimeoutException {
        long abortTime = System.currentTimeMillis() + timeout;
        if (recover && !this.isRecoverable()) {
            throw new IllegalStateException("lock implementation is not recoverable");
        }
        try {
            if (recover) {
                if (!this.execute(new RecoverLockNode(recoveryKey)).booleanValue()) {
                    return null;
                }
            } else {
                this.execute(new CreateLockNode());
            }
            this.execute(new AcquireLockLoop(abortTime, timeout, recover));
            return this.asLockType();
        }
        catch (Exception e) {
            try {
                this.killLock(KillReason.ACQUIRE_FAILED);
            }
            catch (Exception cleanUpException) {
                LOG.error("Error during cleanup of failed lock acquisition. Please check server logs and also check lock service server. The lock may now ne stalled. {}", (Object)ExceptionUtils.getRootCauseMessage((Throwable)cleanUpException));
            }
            if (e instanceof InterruptedException) {
                throw (InterruptedException)e;
            }
            if (e instanceof TimeoutException) {
                throw (TimeoutException)e;
            }
            throw new LockAcquirationFailedException(this.lockId, e);
        }
    }

    T asLockType() {
        return (T)this;
    }

    @Override
    protected void disconnect() {
        this.killLock(KillReason.ZOOKEEPER_DISCONNECT);
    }

    @Override
    protected void doClose() {
        if (CloudDebug.zooKeeperLockService) {
            LOG.debug("Closing lock {}/{}", (Object)this.lockNodePath, (Object)this.myLockName);
        }
        this.activeLockName = null;
    }

    @Override
    public String getId() {
        return this.lockId;
    }

    public final String getMyLockName() {
        return this.myLockName;
    }

    @Override
    protected String getToStringDetails() {
        StringBuilder details = new StringBuilder();
        details.append("id=").append(this.lockId);
        if (this.isValid()) {
            details.append(", ACQUIRED");
        }
        details.append(", lockName=").append(this.myLockName);
        details.append(", activeLockName=").append(this.activeLockName);
        return details.toString();
    }

    protected final boolean isEphemeral() {
        return this.ephemeral;
    }

    protected final boolean isRecoverable() {
        return this.recovarable;
    }

    @Override
    public boolean isValid() {
        String myLockName = this.myLockName;
        String activeLockName = this.activeLockName;
        return myLockName != null && activeLockName != null && activeLockName.equals(myLockName);
    }

    void killLock(KillReason killReason) {
        if (this.myLockName == null || this.isClosed()) {
            return;
        }
        this.close();
        if (CloudDebug.zooKeeperLockService) {
            LOG.debug("Killing lock {}/{}", (Object)this.lockNodePath, (Object)this.myLockName);
        }
        try {
            try {
                if (this.shouldDeleteOnKill(killReason)) {
                    this.execute(new DeleteLockNode());
                }
                this.notifyLockReleased(killReason);
            }
            catch (KeeperException.SessionExpiredException sessionExpiredException) {
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("ZooKeeper session expired. Relying on ZooKeeper server to remove lock node {}/{}", (Object)this.lockNodePath, (Object)this.myLockName);
                }
                this.notifyLockReleased(KillReason.ZOOKEEPER_DISCONNECT);
                this.close();
            }
            catch (Exception e) {
                LOG.warn("Unable to remove lock node {}. Please check server logs and also ZooKeeper. If node still exists and the session is not closed it might never get released. However, it should get released automatically after the session times out on the ZooKeeper server. {}", (Object)this.lockNodePath.append(this.myLockName), (Object)ExceptionUtils.getRootCauseMessage((Throwable)e));
                this.close();
            }
        }
        finally {
            this.close();
        }
    }

    void notifyLockAcquired() {
        LOG.info("Successfully acquired lock {}!", (Object)this.getId());
        if (this.lockMonitor != null && !this.isClosed() && this.isValid()) {
            this.lockMonitor.lockAcquired(this.asLockType());
        }
    }

    void notifyLockReleased(KillReason reason) {
        boolean released;
        boolean bl = released = reason == KillReason.REGULAR_RELEASE;
        if (reason != KillReason.ACQUIRE_FAILED) {
            LOG.info(released ? "Successfully released lock {}!" : "Lost lock {}!", (Object)this.getId());
        }
        if (this.lockMonitor != null) {
            if (released) {
                this.lockMonitor.lockReleased(this.asLockType());
            } else {
                this.lockMonitor.lockLost(this.asLockType());
            }
        }
    }

    @Override
    public void release() {
        this.killLock(KillReason.REGULAR_RELEASE);
    }

    private boolean shouldDeleteOnKill(KillReason killReason) {
        switch (killReason) {
            case LOCK_DELETED: 
            case LOCK_STOLEN: {
                return false;
            }
            case REGULAR_RELEASE: {
                return true;
            }
            case ZOOKEEPER_DISCONNECT: {
                return !this.isRecoverable();
            }
            case ACQUIRE_FAILED: {
                return true;
            }
        }
        LOG.warn("Unhandled lock kill reason {}. Please report this issue to the developers. They should sanity check the implementation.", (Object)killReason);
        return true;
    }

    private final class AcquireLockLoop
    implements Callable<Boolean> {
        private final long abortTime;
        private final long timeout;
        private final boolean recover;

        private AcquireLockLoop(long abortTime, long timeout, boolean recover) {
            this.abortTime = abortTime;
            this.timeout = timeout;
            this.recover = recover;
        }

        @Override
        public Boolean call() throws Exception {
            ZooKeeperGate zk = ZooKeeperGate.get();
            do {
                Object[] nodeNames;
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Starting acquire lock loop for lock {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)ZooKeeperLock.this.myLockName);
                }
                if ((nodeNames = zk.readChildrenNames(ZooKeeperLock.this.lockNodePath, null).toArray()).length == 0) {
                    LOG.warn("Unexpected child count for ZooKeeper node {}. We just created a sequential child but it wasn't there. This may indicate an instability in the system.", (Object)ZooKeeperLock.this.lockNodePath);
                    continue;
                }
                Arrays.sort(nodeNames, new Comparator<Object>(){

                    @Override
                    public int compare(Object o1, Object o2) {
                        String n1 = (String)o1;
                        String n2 = (String)o2;
                        int sequence1 = ZooKeeperLock.getSequenceNumber(n1);
                        int sequence2 = ZooKeeperLock.getSequenceNumber(n2);
                        if (sequence1 == -1) {
                            return sequence2 != -1 ? 1 : n1.compareTo(n2);
                        }
                        return sequence2 == -1 ? -1 : sequence1 - sequence2;
                    }
                });
                ZooKeeperLock.this.activeLockName = (String)nodeNames[0];
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Found active lock {} for lock {}", (Object)ZooKeeperLock.this.activeLockName, (Object)ZooKeeperLock.this.lockNodePath);
                }
                if (ZooKeeperLock.this.isValid()) {
                    ZooKeeperLock.this.notifyLockAcquired();
                    return true;
                }
                String precedingNodeName = null;
                int i = 0;
                while (i < nodeNames.length) {
                    if (ZooKeeperLock.this.myLockName.equals(nodeNames[i])) {
                        precedingNodeName = (String)nodeNames[i - 1];
                    }
                    ++i;
                }
                if (precedingNodeName == null) {
                    if (this.recover) {
                        throw new LockAcquirationFailedException(ZooKeeperLock.this.getId(), "Impossible to recover lock. The preceding lock could not be discovered.");
                    }
                    throw new LockAcquirationFailedException(ZooKeeperLock.this.getId(), "Impossible to acquire lock. The preceding lock could not be discovered.");
                }
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Found preceding lock {} for lock {}", precedingNodeName, (Object)ZooKeeperLock.this.lockNodePath);
                }
                WaitForDeletionMonitor waitForDeletionMonitor = new WaitForDeletionMonitor();
                if (zk.exists(ZooKeeperLock.this.lockNodePath.append(precedingNodeName), waitForDeletionMonitor)) {
                    if (CloudDebug.zooKeeperLockService) {
                        LOG.debug("Waiting for preceeing lock {} to release lock {}", (Object)precedingNodeName, (Object)ZooKeeperLock.this.lockNodePath);
                    }
                    if (!waitForDeletionMonitor.await(this.timeout)) {
                        if (CloudDebug.zooKeeperLockService) {
                            LOG.debug("Timeout waiting for preceeing lock {} to release lock {}", (Object)precedingNodeName, (Object)ZooKeeperLock.this.lockNodePath);
                        }
                        throw new TimeoutException(String.format("Unable to acquire lock %s within the given timeout.", ZooKeeperLock.this.getId()));
                    }
                    if (CloudDebug.zooKeeperLockService) {
                        LOG.debug("Preceeing lock {} released lock {}", (Object)precedingNodeName, (Object)ZooKeeperLock.this.lockNodePath);
                    }
                }
                if (!CloudDebug.zooKeeperLockService) continue;
                LOG.debug("End acquire lock loop for lock {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)ZooKeeperLock.this.myLockName);
            } while (this.timeout <= 0L || this.abortTime > System.currentTimeMillis());
            if (CloudDebug.zooKeeperLockService) {
                LOG.debug("Timeout retrying to acquire lock {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)ZooKeeperLock.this.myLockName);
            }
            throw new TimeoutException(String.format("Unable to acquire lock %s within the given timeout.", ZooKeeperLock.this.getId()));
        }
    }

    private final class CreateLockNode
    implements Callable<Boolean> {
        private CreateLockNode() {
        }

        @Override
        public Boolean call() throws Exception {
            if (!ZooKeeperLock.this.isClosed() && ZooKeeperLock.this.myLockName == null) {
                ZooKeeperGate zk = ZooKeeperGate.get();
                IPath nodePath = zk.createPath(ZooKeeperLock.this.lockNodePath.append(ZooKeeperLock.LOCK_NAME_PREFIX), ZooKeeperLock.this.isEphemeral() ? CreateMode.EPHEMERAL_SEQUENTIAL : CreateMode.PERSISTENT_SEQUENTIAL, ZooKeeperLock.this.lockNodeContent);
                ZooKeeperLock.this.myLockName = nodePath.lastSegment();
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Created lock node {} for lock {}", (Object)ZooKeeperLock.this.myLockName, (Object)ZooKeeperLock.this.lockNodePath);
                }
                ZooKeeperLock.this.myRecoveryKey = ZooKeeperLock.createRecoveryKey(ZooKeeperLock.this.myLockName, ZooKeeperLock.this.lockNodeContent);
                zk.readRecord(nodePath, ZooKeeperLock.this.killMonitor, null);
            }
            return true;
        }
    }

    private final class DeleteLockNode
    implements Callable<Boolean> {
        private DeleteLockNode() {
        }

        @Override
        public Boolean call() throws Exception {
            block4: {
                String lockName = ZooKeeperLock.this.myLockName;
                if (lockName == null) {
                    return false;
                }
                try {
                    if (CloudDebug.zooKeeperLockService) {
                        LOG.debug("Deleting lock node in ZooKeeper {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)ZooKeeperLock.this.myLockName);
                    }
                    ZooKeeperGate.get().deletePath(ZooKeeperLock.this.lockNodePath.append(lockName), -1);
                }
                catch (KeeperException.NoNodeException noNodeException) {
                    if (!CloudDebug.zooKeeperLockService) break block4;
                    LOG.debug("Lock node already gone {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)ZooKeeperLock.this.myLockName);
                }
            }
            ZooKeeperLock.this.myLockName = null;
            return true;
        }
    }

    static enum KillReason {
        ZOOKEEPER_DISCONNECT,
        LOCK_DELETED,
        LOCK_STOLEN,
        REGULAR_RELEASE,
        ACQUIRE_FAILED;

    }

    private final class RecoverLockNode
    implements Callable<Boolean> {
        private final String lockName;
        private final String expectedNodeContent;

        public RecoverLockNode(String recoveryKey) {
            if (StringUtils.isBlank((String)recoveryKey)) {
                throw new IllegalArgumentException("recovery key must not be empty");
            }
            String[] extractRecoveryKeyDetails = ZooKeeperLock.extractRecoveryKeyDetails(recoveryKey);
            this.lockName = extractRecoveryKeyDetails[0];
            this.expectedNodeContent = extractRecoveryKeyDetails[1];
        }

        @Override
        public Boolean call() throws Exception {
            if (!ZooKeeperLock.this.isClosed() && ZooKeeperLock.this.myLockName == null) {
                Stat stat;
                IPath nodePath;
                String record;
                ZooKeeperGate zk = ZooKeeperGate.get();
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Recovery attempt for lock node {} for lock {}", (Object)this.lockName, (Object)ZooKeeperLock.this.lockNodePath);
                }
                if (StringUtils.isBlank((String)(record = zk.readRecord(nodePath = ZooKeeperLock.this.lockNodePath.append(this.lockName), "", stat = new Stat())))) {
                    if (CloudDebug.zooKeeperLockService) {
                        LOG.debug("Recovery attempt failed. Lock node {}/{} does not exists", (Object)ZooKeeperLock.this.lockNodePath, (Object)this.lockName);
                    }
                    return false;
                }
                if (!StringUtils.equals((String)record, (String)this.expectedNodeContent)) {
                    if (CloudDebug.zooKeeperLockService) {
                        LOG.debug("Recovery attempt failed. Recovery key does not match for lock node {}/{}", (Object)ZooKeeperLock.this.lockNodePath, (Object)this.lockName);
                    }
                    throw new LockAcquirationFailedException(ZooKeeperLock.this.lockId, "Unable to recover lock. The recovery key does not match.");
                }
                ZooKeeperLock.this.myLockName = nodePath.lastSegment();
                if (CloudDebug.zooKeeperLockService) {
                    LOG.debug("Recovered lock node {} for lock {}", (Object)ZooKeeperLock.this.myLockName, (Object)ZooKeeperLock.this.lockNodePath);
                }
                ZooKeeperLock.this.myRecoveryKey = ZooKeeperLock.createRecoveryKey(ZooKeeperLock.this.myLockName, ZooKeeperLock.this.lockNodeContent);
                zk.writeRecord(nodePath, ZooKeeperLock.this.myRecoveryKey, stat.getVersion());
                zk.readRecord(nodePath, ZooKeeperLock.this.killMonitor, null);
            }
            return true;
        }
    }

    private static class WaitForDeletionMonitor
    extends ZooKeeperMonitor {
        private static CountDownLatch deletionHappend = new CountDownLatch(1);

        private WaitForDeletionMonitor() {
        }

        public boolean await(long timeout) throws InterruptedException {
            if (timeout > 0L) {
                return deletionHappend.await(timeout, TimeUnit.MILLISECONDS);
            }
            deletionHappend.await();
            return true;
        }

        @Override
        protected void pathDeleted(String path) {
            deletionHappend.countDown();
        }
    }
}

