/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.rt.server.scheduler;

import org.eclipse.scout.commons.annotations.ConfigOperation;
import org.eclipse.scout.commons.annotations.Order;
import org.eclipse.scout.commons.exception.ProcessingException;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;

public abstract class AbstractSchedulerJob implements ISchedulerJob {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(AbstractSchedulerJob.class);

  private final String m_groupId;
  private final String m_jobId;
  private volatile boolean m_disposed;
  private volatile boolean m_interrupted;

  public AbstractSchedulerJob(String groupId, String jobId) {
    m_groupId = groupId;
    m_jobId = jobId;
  }

  public String getGroupId() {
    return m_groupId;
  }

  public String getJobId() {
    return m_jobId;
  }

  public boolean acceptTick(TickSignal signal) {
    if (LOG.isDebugEnabled()) LOG.debug("task " + getJobId() + " at " + signal);
    int second = signal.getSecond();
    int minute = signal.getMinute();
    int hour = signal.getHour();
    int day = signal.getDay();
    int week = signal.getWeek();
    int month = signal.getMonth();
    int year = signal.getYear();
    int dayOfWeek = signal.getDayOfWeek();
    int dayOfMonthReverse = signal.getDayOfMonthReverse();
    int dayOfYear = signal.getDayOfYear();
    int secondOfDay = signal.getSecondOfDay();
    boolean accepted = execAcceptTick(signal, second, minute, hour, day, week, month, year, dayOfWeek, dayOfMonthReverse, dayOfYear, secondOfDay);
    if (accepted) {
      if (LOG.isInfoEnabled()) LOG.info("accepted task " + getJobId() + " triggered at " + signal);
    }
    return accepted;
  }

  @ConfigOperation
  @Order(20)
  public void run(IScheduler scheduler, TickSignal signal) throws ProcessingException {
  }

  @ConfigOperation
  @Order(10)
  protected boolean execAcceptTick(TickSignal signal, int second, int minute, int hour, int day, int week, int month, int year, int dayOfWeek, int dayOfMonthReverse, int dayOfYear, int secondOfDay) {
    return false;
  }

  public boolean isInterrupted() {
    return m_interrupted;
  }

  public void setInterrupted(boolean b) {
    m_interrupted = b;
  }

  public boolean isDisposed() {
    return m_disposed;
  }

  public void setDisposed(boolean b) {
    m_disposed = b;
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "[" + m_groupId + "." + m_jobId + "]";
  }

}
