/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.commons;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.TreeMap;

import org.eclipse.scout.commons.annotations.IOrdered;
import org.eclipse.scout.commons.annotations.InjectFieldTo;
import org.eclipse.scout.commons.annotations.Order;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;

/**
 * Configuration-related utilities.
 */
public final class ConfigurationUtility {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(ConfigurationUtility.class);

  private ConfigurationUtility() {
  }

  /**
   * Filters the given class array and sorts the remaining elements according to
   * their {@link Order} annotation.
   * <p>
   * By default, the method throws an {@link IllegalArgumentException} if one of the remaining classes is not annotated
   * by {@link Order}. The behavior can be switched off by setting the system property
   * <code>bsi.debug.innerclass.order</code> to an arbitrary value.
   * 
   * @param classes
   * @param filter
   * @return
   * @throws IllegalArgumentException
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T>[] sortFilteredClassesByOrderAnnotation(Class[] classes, Class<T> filter) {
    TreeMap<CompositeObject, Class> orderedClassesMap = new TreeMap<CompositeObject, Class>();
    for (int i = 0; i < classes.length; i++) {
      if (filter.isAssignableFrom(classes[i])) {
        if (classes[i].isAnnotationPresent(Order.class)) {
          Order order = (Order) classes[i].getAnnotation(Order.class);
          orderedClassesMap.put(new CompositeObject(order.value(), i), classes[i]);
        }
        else {
          LOG.error("missing @Order annotation: " + classes[i].getName());
          orderedClassesMap.put(new CompositeObject(Double.MAX_VALUE, i), classes[i]);
        }
      }
    }
    return orderedClassesMap.values().toArray(new Class[orderedClassesMap.size()]);
  }

  /**
   * @deprecated use {@link #sortByOrder(Collection)} instead.
   */
  @Deprecated
  public static <T> Collection<T> sortByOrderAnnotation(Collection<T> list) {
    return sortByOrder(list);
  }

  /**
   * Sorts the elements according to their order:
   * <ol>
   * <li>If an {@link Order} annotation is available, its {@link Order#value()} is used</li>
   * <li>If the object implements {@link IOrdered}, {@link IOrdered#getOrder()} is used.</li>
   * <li>Finally, the index in the original collection is used</li>
   * </ol>
   * 
   * @since 3.8.1
   */
  public static <T> Collection<T> sortByOrder(Collection<T> list) {
    if (list == null) {
      return null;
    }
    TreeMap<CompositeObject, T> sortMap = new TreeMap<CompositeObject, T>();
    int index = 0;
    for (T element : list) {
      Class<?> c = element.getClass();
      double order;
      if (c.isAnnotationPresent(Order.class)) {
        order = c.getAnnotation(Order.class).value();
      }
      else if (element instanceof IOrdered) {
        order = ((IOrdered) element).getOrder();
      }
      else {
        order = (double) index;
      }
      sortMap.put(new CompositeObject(order, index), element);
      index++;
    }
    return sortMap.values();
  }

  /**
   * Filters the given class array and returns the first occurrence of an
   * instantiable class of filter
   * 
   * @param classes
   * @param filter
   * @return first occurrence of filter, might be annotated with {@link InjectFieldTo}
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T> filterClass(Class[] classes, Class<T> filter) {
    for (Class c : classes) {
      if (filter.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
        return c;
      }
    }
    return null;
  }

  /**
   * same as {@link #filterClass(Class[], Class)} but ignoring classes with {@link InjectFieldTo} annotation
   * 
   * @since 3.8.1
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T> filterClassIgnoringInjectFieldAnnotation(Class[] classes, Class<T> filter) {
    for (Class c : classes) {
      if (filter.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
        if (!c.isAnnotationPresent(InjectFieldTo.class)) {
          return c;
        }
      }
    }
    return null;
  }

  /**
   * Filters the given class array and returns all occurrences of instantiable
   * classes of filter
   * 
   * @param classes
   * @param filter
   * @return all occurrences of filter
   * @since 3.8.1
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T>[] filterClasses(Class[] classes, Class<T> filter) {
    ArrayList<Class<T>> list = new ArrayList<Class<T>>();
    for (Class c : classes) {
      if (filter.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
        list.add(c);
      }
    }
    return list.toArray(new Class[0]);
  }

  /**
   * same as {@link #filterClasses(Class[], Class)} but ignoring classes with {@link InjectFieldTo} annotation
   * 
   * @since 3.8.1
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T>[] filterClassesIgnoringInjectFieldAnnotation(Class[] classes, Class<T> filter) {
    ArrayList<Class<T>> list = new ArrayList<Class<T>>();
    for (Class c : classes) {
      if (filter.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
        if (!c.isAnnotationPresent(InjectFieldTo.class)) {
          list.add(c);
        }
      }
    }
    return list.toArray(new Class[0]);
  }

  /**
   * same as {@link #filterClasses(Class[], Class)} but only accepting classes with {@link InjectFieldTo} annotation
   * 
   * @since 3.8.1
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T>[] filterClassesWithInjectFieldAnnotation(Class[] classes, Class<T> filter) {
    ArrayList<Class<T>> list = new ArrayList<Class<T>>();
    for (Class c : classes) {
      if (filter.isAssignableFrom(c) && !Modifier.isAbstract(c.getModifiers())) {
        if (c.isAnnotationPresent(InjectFieldTo.class)) {
          list.add(c);
        }
      }
    }
    return list.toArray(new Class[0]);
  }

  /**
   * get all declared classes (inner types) of the specified class and all its
   * super classes
   */
  public static Class[] getDeclaredPublicClasses(Class c) {
    return c.getClasses();
  }

  public static <T> T newInnerInstance(Object instance, Class<T> innerClass) throws Exception {
    if (innerClass.getDeclaringClass() != null && (innerClass.getModifiers() & Modifier.STATIC) == 0) {
      Constructor<T> c = innerClass.getDeclaredConstructor(new Class[]{innerClass.getDeclaringClass()});
      return c.newInstance(new Object[]{instance});
    }
    else {
      return innerClass.newInstance();
    }
  }

  /**
   * @return true if the declared method is overwritten in implementationType
   */
  public static boolean isMethodOverwrite(Class<?> declaringType, String methodName, Class[] parameterTypes, Class<?> implementationType) {
    try {
      Method declaredMethod;
      try {
        declaredMethod = declaringType.getDeclaredMethod(methodName, parameterTypes);
      }
      catch (NoSuchMethodException e) {
        LOG.error("cannot find declared method " + declaringType.getName() + "." + methodName, e);
        return false;
      }
      Class<?> c = implementationType;
      while (c != null && c != declaringType) {
        try {
          //check if method is avaliable
          c.getDeclaredMethod(declaredMethod.getName(), declaredMethod.getParameterTypes());
          return true;
        }
        catch (NoSuchMethodException e) {
          //nop
        }
        //up
        c = c.getSuperclass();
      }
    }
    catch (Throwable t) {
      LOG.error("declaringType=" + declaringType + ", methodName=" + methodName + ", parameterTypes=" + parameterTypes + ", implementationType=" + implementationType, t);
    }
    return false;
  }

  /**
   * @return Returns the given objects enclosing container type, i.e the first class on the enclosing classes path that
   *         is abstract or the outermost enclosing class. The latter is the primary type.
   */
  public static Class<?> getEnclosingContainerType(Object o) {
    if (o == null) {
      return null;
    }
    Class<?> c = o.getClass();
    while (!Modifier.isAbstract(c.getModifiers()) && c.getEnclosingClass() != null) {
      c = c.getEnclosingClass();
    }
    return c;
  }
}
