/*******************************************************************************
 * Copyright (c) 2010 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Gregoire Dupe (Mia-Software)
 *     Nicolas Bros (Mia-Software)
 *******************************************************************************/
package org.eclipse.emf.facet.query.java.ui.internal.wizard.page;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Manifest;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.codegen.ecore.genmodel.GenModel;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.facet.efacet.FacetSet;
import org.eclipse.emf.facet.query.java.core.IJavaQuery;
import org.eclipse.emf.facet.query.java.ui.internal.Activator;
import org.eclipse.emf.facet.query.java.ui.internal.Messages;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.util.core.internal.JavaUtils;
import org.eclipse.emf.facet.util.core.internal.PluginUtils;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.ManifestElement;

public class NewQueryClassWizardPage extends NewClassWizardPage implements IQueryAttributesWizardPage {

	// This class has been copied from
	// org.eclipse.emf.facet.infra.query.ui.wizards.NewQueryClassWizardPage

	public NewQueryClassWizardPage() {
	}

	public void init(final FacetSet modelQuerySet, final String facetName, final EClassifier eType) {
		IJavaProject javaProject = findJavaProject(modelQuerySet);

		List<String> interfaceList = new ArrayList<String>();
		interfaceList.add(IJavaQuery.class.getName());
		setSuperInterfaces(interfaceList, false);

		String packageName = javaProject.getProject().getName() + ".queries." + eType.getName().toLowerCase(); //$NON-NLS-1$
		setTypeName(facetName, true);
		selectPackage(javaProject, packageName);
		setModifiers(this.F_PUBLIC, false);
		setSuperClass("", false); //$NON-NLS-1$
		setEnclosingTypeSelection(false, false);
		setAddComments(false, false);
	}

	private void selectPackage(final IJavaProject javaProject, final String packageName) {
		try {
			for (IPackageFragmentRoot packageFragmentRootToTest : javaProject.getPackageFragmentRoots()) {
				if (packageFragmentRootToTest.getKind() == IPackageFragmentRoot.K_SOURCE) {
					IPackageFragmentRoot packageFragmentRoot = packageFragmentRootToTest;
					setPackageFragmentRoot(packageFragmentRoot, false);
					IPackageFragment packageFragment = packageFragmentRoot.getPackageFragment(packageName);
					setPackageFragment(packageFragment, true);
					break;
				}
			}
		} catch (JavaModelException e1) {
			Logger.logError(e1, Activator.getDefault());
		}
	}

	private static IJavaProject findJavaProject(final FacetSet modelQuerySet) {
		Resource resource = modelQuerySet.eResource();
		URI uri = resource.getURI();
		if (uri.isPlatformResource()) {
			String projectName = uri.segment(1);
			IWorkspace ws = ResourcesPlugin.getWorkspace();
			IProject project = ws.getRoot().getProject(projectName);
			return JavaCore.create(project);
		}
		return null;
	}

	public void apply(final EClassifier returnType, final int upperBound, final EClassifier scope) {
		try {
			ICompilationUnit javaClass = createJavaClass(returnType, upperBound, scope);
			IProject project = javaClass.getCorrespondingResource().getProject();
			setupProject(project, returnType, scope);

		} catch (Exception e) {
			Logger.logError(e, Activator.getDefault());
			MessageDialog.openError(getShell(), Messages.NewQueryClassWizardPage_Failed_to_create_java_class,
					Messages.NewQueryClassWizardPage_Failed_to_create_query_implementation);
		}
	}

	/**
	 * Setup the given project's Manifest so that metamodels referenced by the query's <code>returnType</code> and
	 * <code>scope</code> are available on the classpath.
	 */
	private static void setupProject(final IProject project, final EClassifier returnType, final EClassifier scope) {
		try {
			String[] requiredBundles = findRequiredBundles(returnType, scope);
			// start with all and then remove those already present
			List<String> missingRequiredBundles = new ArrayList<String>();
			for (String requiredBundle : requiredBundles) {
				missingRequiredBundles.add(requiredBundle);
			}
			missingRequiredBundles.add("org.eclipse.emf.facet.efacet.core"); //$NON-NLS-1$
			missingRequiredBundles.add("org.eclipse.emf.facet.query.java.core"); //$NON-NLS-1$

			PluginUtils.configureAsPluginProject(project);
			IFile manifestResource = (IFile) project.findMember(new Path("/META-INF/MANIFEST.MF")); //$NON-NLS-1$
			manifestResource.refreshLocal(IResource.DEPTH_ONE, new NullProgressMonitor());
			InputStream contents = manifestResource.getContents();
			Manifest manifest = new Manifest(contents);
			String requires = manifest.getMainAttributes().getValue("Require-Bundle"); //$NON-NLS-1$
			if (requires != null) {
				ManifestElement[] manifestElements = ManifestElement.parseHeader("Require-Bundle", requires); //$NON-NLS-1$
				for (ManifestElement manifestElement : manifestElements) {
					// make sure it won't be duplicated
					missingRequiredBundles.remove(manifestElement.getValue());
				}
			}

			StringBuilder newRequires = new StringBuilder();
			if (requires != null) {
				newRequires.append(requires);
			}
			for (int i = 0; i < missingRequiredBundles.size(); i++) {
				String missingRequiredBundle = missingRequiredBundles.get(i);
				if (i != 0 || requires != null) {
					newRequires.append(","); //$NON-NLS-1$
				}
				newRequires.append(missingRequiredBundle);
			}

			manifest.getMainAttributes().putValue("Require-Bundle", newRequires.toString()); //$NON-NLS-1$

			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			manifest.write(outputStream);
			ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(outputStream.toByteArray());
			manifestResource.setContents(byteArrayInputStream, true, true, new NullProgressMonitor());
		} catch (Exception e) {
			Logger.logError(e, Activator.getDefault());
		}
	}

	/**
	 * Find the bundles containing the implementations of the metamodels referenced by the given classifiers.
	 */
	private static String[] findRequiredBundles(final EClassifier returnType, final EClassifier scope) {
		Set<String> requiredBundles = new HashSet<String>();
		Set<EPackage> requiredEPackages = new HashSet<EPackage>();
		if (returnType != null) {
			requiredEPackages.add(returnType.getEPackage());
		}
		if (scope != null) {
			requiredEPackages.add(scope.getEPackage());
		}

		Map<String, URI> genModelLocationMap = EcorePlugin.getEPackageNsURIToGenModelLocationMap();

		for (EPackage ePackage : requiredEPackages) {
			try {
				URI genModelURI = genModelLocationMap.get(ePackage.getNsURI());
				if (genModelURI != null) {
					ResourceSet resourceSet = new ResourceSetImpl();
					Resource resource = resourceSet.createResource(genModelURI);
					resource.load(Collections.emptyMap());
					GenModel genModel = (GenModel) resource.getContents().get(0);
					String modelPluginID = genModel.getModelPluginID();
					if (modelPluginID != null) {
						requiredBundles.add(modelPluginID);
					}
				} else {
					Logger.logWarning("Couldn't add the metamodel implementation plug-in" //$NON-NLS-1$
							+ " to the dependencies automatically because" //$NON-NLS-1$
							+ " the corresponding genmodel couldn't be found: " //$NON-NLS-1$
							+ ePackage.getNsURI(), Activator.getDefault());
				}
			} catch (Exception e) {
				Logger.logError(e, Activator.getDefault());
			}
		}
		return requiredBundles.toArray(new String[requiredBundles.size()]);
	}

	private ICompilationUnit createJavaClass(final EClassifier returnTypeClass, final int upperBound, final EClassifier scope)
			throws JavaModelException {
		IPackageFragmentRoot root = getPackageFragmentRoot();
		IPackageFragment packageFragment = getPackageFragment();
		if (packageFragment == null) {
			packageFragment = root.getPackageFragment(""); //$NON-NLS-1$
		}

		String packageName = packageFragment.getElementName();
		if (!packageFragment.exists()) {
			packageFragment = root.createPackageFragment(packageName, true, new NullProgressMonitor());
		}

		String typeName = getTypeName();

		StringBuilder contents = new StringBuilder();
		createJavaClassContents(contents, typeName, packageName, returnTypeClass, upperBound, scope);

		String cuName = getCompilationUnitName(typeName);
		ICompilationUnit compilationUnit = packageFragment.createCompilationUnit(cuName, contents.toString(), false, new NullProgressMonitor());

		return compilationUnit;
	}

	private static void createJavaClassContents(final StringBuilder contents, final String name, final String packageName,
			final EClassifier returnTypeClass, final int upperBound, final EClassifier scope) {
		if (packageName.length() > 0) {
			contents.append("package "); //$NON-NLS-1$
			contents.append(packageName);
			contents.append(";\n\n"); //$NON-NLS-1$
		}
		boolean importCollection = false;

		String returnType;
		String shortReturnType;
		if (returnTypeClass != null) {
			returnType = returnTypeClass.getInstanceClassName();
			returnType = JavaUtils.objectType(returnType);
			int lastDotPos = returnType.lastIndexOf('.');
			if (upperBound != 1) {
				importCollection = true;
				shortReturnType = "Collection<" + returnType.substring(lastDotPos + 1) + ">"; //$NON-NLS-1$//$NON-NLS-2$
			} else {
				shortReturnType = returnType.substring(lastDotPos + 1);
			}
		} else {
			returnType = "java.lang.Object"; //$NON-NLS-1$
			shortReturnType = "Object"; //$NON-NLS-1$
		}

		// EList<EClass> scope = modelQuery.getScope();
		String scopeType;
		String shortScopeType;
		// if (scope != null && scope.size() == 1) {
		if (scope != null) {
			scopeType = scope.getInstanceClassName();
			scopeType = JavaUtils.objectType(scopeType);
			shortScopeType = scopeType.substring(scopeType.lastIndexOf('.') + 1);
		} else {
			scopeType = "org.eclipse.emf.ecore.EObject"; //$NON-NLS-1$
			shortScopeType = "EObject"; //$NON-NLS-1$
		}

		if (importCollection) {
			contents.append("import java.util.Collection;\n"); //$NON-NLS-1$
		}
		if (mustImport(returnType)) {
			contents.append("import " + returnType + ";\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (!returnType.equals(scopeType) && mustImport(scopeType)) {
			contents.append("import " + scopeType + ";\n"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		contents.append("import org.eclipse.emf.facet.efacet.core.exception.QueryExecutionException;\n"); //$NON-NLS-1$
		contents.append("import org.eclipse.emf.facet.query.java.core.IJavaQuery;\n"); //$NON-NLS-1$
		contents.append("import org.eclipse.emf.facet.query.java.core.IParameterValueList;\n"); //$NON-NLS-1$
		contents.append("\n"); //$NON-NLS-1$
		contents.append("public class " + name + " implements IJavaQuery<" + shortScopeType + ", " + shortReturnType + "> {\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		contents.append("	public " + shortReturnType + " evaluate(final " + shortScopeType + " context, final IParameterValueList parameterValues)\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		contents.append("			throws QueryExecutionException {\n"); //$NON-NLS-1$
		contents.append("		// TODO Auto-generated method stub\n"); //$NON-NLS-1$
		contents.append("		return null;\n"); //$NON-NLS-1$
		contents.append("	}\n"); //$NON-NLS-1$
		contents.append("}\n"); //$NON-NLS-1$
	}

	private static boolean mustImport(final String type) {
		return type.contains(".") && !type.startsWith("java.lang."); //$NON-NLS-1$ //$NON-NLS-2$
	}
}
