/**
 * <copyright>
 *
 * Copyright (c) 2007, 2008 IBM Corporation, Zeligsoft Inc., and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *   Zeligsoft - Bug 253252
 *
 * </copyright>
 *
 * $Id: EvaluationOptions.java,v 1.2 2008/11/05 16:30:17 cdamus Exp $
 */

package org.eclipse.ocl.options;

import org.eclipse.ocl.EvaluationEnvironment;
import org.eclipse.ocl.util.OCLUtil;


/**
 * Options applicable to {@link EvaluationEnvironment}s to
 * {@linkplain Customizable customize} their evaluation behaviour.
 * 
 * @author Christian W. Damus (cdamus)
 * 
 * @since 1.2
 */
public class EvaluationOptions {

    /**
     * <p>
     * Evaluation option indicating whether to implement lax handling of null
     * values in certain <tt>OclAny</tt>-defined operations.  When <tt>true</tt>,
     * the <tt>null.oclIsKindOf(OclType)</tt> and <tt>null.oclIsTypeOf(OclType)</tt>
     * operations will return <tt>true</tt> for any OCL type instead of
     * returning <tt>OclInvalid</tt>, as <tt>OclVoid</tt> is defined as
     * conforming to all other types.  Similarly, <tt>null.oclAsType(OclType)</tt>
     * will return <tt>null</tt> for any OCL type instead of <tt>OclInvalid</tt>
     * as prescribed by the OCL 2.0 Specification.
     * </p><p>
     * For backward compatibility with the 1.1 release behaviour, the default
     * value of this option is <tt>true</tt>.  For strict conformance to the
     * specification, use <tt>false</tt>.
     * </p>
     * 
     * @since 1.2
     */
    public static final Option<Boolean> LAX_NULL_HANDLING = new BooleanOption(
    		OCLUtil.PLUGIN_ID, "lax.null.handling", true); //$NON-NLS-1$

    /**
     * <p>
     * Evaluation option indicating whether to perform dynamic source-type dependent
     * dispatch of operation calls.
     * </p><p>
     * For backward compatibility, the default value of this option is <tt>false</tt>. 
     * For Object Oriented behaviour use <tt>true</tt>.
     * </p><p>
     * Prior to OCL 2.5, when Object Oriented behaviour may be specified explicitly, it is
     * debateable as to which behaviour is specification compliant.
     * </p><p>
     * To avoid a major performance degradation, the dynamic dispatcher uses a cache to
     * avoid repeated searches for operations in the inheritance tree. The consequent cost
     * of dynamic dispatch may therefore be a few percent, If the parsing option to use
     * type caches is enabled, the evaluator may reuse the analysis cache and avoid some
     * startup costs.
     * </p>
     * 
     * @since 3.2
     */
    public static final Option<Boolean> DYNAMIC_DISPATCH = new BooleanOption(
    		OCLUtil.PLUGIN_ID, "dynamic.dispatch", false); //$NON-NLS-1$

    /**
     * Not instantiable by clients.
     */
    private EvaluationOptions() {
        super();
    }

    /**
     * Add an option to apply to the specified environment, adapting it as
     * necessary to the {@link Customizable} API.
     * 
     * @param env an evaluation environment on which to set an option
     * @param option the option
     * @param value the option's value
     * 
     * @see Cusotmizable#setOption(Option, Object)
     */
    public static <T> void setOption(EvaluationEnvironment<?, ?, ?, ?, ?> env,
            Option<T> option, T value) {
        
        Customizable custom = OCLUtil.getAdapter(env, Customizable.class);
        if (custom != null) {
            custom.setOption(option, value);
        }
    }

    /**
     * Obtains the value of the specified option's setting in the the given
     * environment's options map, adapting the environment as necessary to the
     * {@link Customizable} API.  If not already set, return the option's
     * {@linkplain #getDefaultValue() default value}.
     * 
     * @param env an evaluation environment on which to query an option
     * @param option an option to query
     * 
     * @return value of the option
     * 
     * @see Customizable#getValue(Option)
     */
    public static <T> T getValue(EvaluationEnvironment<?, ?, ?, ?, ?> env,
            Option<T> option) {
        
        Customizable custom = OCLUtil.getAdapter(env, Customizable.class);
        if (custom != null) {
            return custom.getValue(option);
        }
        
        return option.getDefaultValue();
    }

}
