/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.gyrex.eventbus.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.eclipse.gyrex.cloud.services.events.EventMessage;
import org.eclipse.gyrex.eventbus.IEventDeserializer;
import org.eclipse.gyrex.eventbus.IEventSerializer;
import org.eclipse.gyrex.eventbus.ITopic;
import org.eclipse.gyrex.eventbus.internal.EventHandler;
import org.eclipse.gyrex.eventbus.internal.EventService;
import org.eclipse.gyrex.eventbus.internal.ReflectionService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Topic
implements ITopic {
    private static final Logger LOG = LoggerFactory.getLogger(Topic.class);
    private final String id;
    private final EventService eventService;
    private final ConcurrentMap<Class<?>, IEventSerializer<Object>> serializersByEventType;
    private final Multimap<String, IEventDeserializer<Object>> deserializersByEventTypeClassName = HashMultimap.create();
    private final Multimap<Class<?>, EventHandler> eventHandlersByType = HashMultimap.create();
    private final ReadWriteLock eventHandlersByTypeLock = new ReentrantReadWriteLock();
    private final AtomicBoolean active = new AtomicBoolean(false);
    private volatile boolean closed = false;

    Topic(String id, Map<String, Object> properties, Map<Class<?>, IEventSerializer<Object>> serializers, Map<Class<?>, IEventDeserializer<Object>> deserializers, EventService eventService) {
        this.id = id;
        this.serializersByEventType = new ConcurrentHashMap(serializers);
        for (Map.Entry<Class<?>, IEventDeserializer<Object>> e : deserializers.entrySet()) {
            this.deserializersByEventTypeClassName.put((Object)e.getKey().getName(), e.getValue());
        }
        this.eventService = eventService;
    }

    private void activateIfNecessary() {
        this.eventHandlersByTypeLock.readLock().lock();
        try {
            if (this.eventHandlersByType.size() > 0 && this.active.compareAndSet(false, true)) {
                this.getEventService().activateTopic(this);
            }
        }
        finally {
            this.eventHandlersByTypeLock.readLock().unlock();
        }
    }

    private void checkClosed() {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"closed");
    }

    @Override
    public void close() {
        this.eventHandlersByTypeLock.writeLock().lock();
        try {
            this.closed = true;
        }
        finally {
            this.eventHandlersByTypeLock.writeLock().unlock();
        }
        if (this.active.compareAndSet(true, false)) {
            this.getEventService().deactivateTopic(this);
        }
        this.eventHandlersByType.clear();
    }

    @VisibleForTesting
    EventMessage createEventMessage(Class<?> eventType, byte[] serializedEvent) {
        return new EventMessage(this.getEventService().newEventId(), eventType.getName(), ByteBuffer.wrap(serializedEvent));
    }

    private void deactivateIfPossible() {
        this.eventHandlersByTypeLock.readLock().lock();
        try {
            if (this.eventHandlersByType.isEmpty() && this.active.compareAndSet(true, false)) {
                this.getEventService().deactivateTopic(this);
            }
        }
        finally {
            this.eventHandlersByTypeLock.readLock().unlock();
        }
    }

    public void dispatchEvent(EventMessage eventMessage) {
        if (this.closed) {
            LOG.trace("Ignoring event message ({}) for topic ({}). Topic is closed.", (Object)eventMessage, (Object)this);
            return;
        }
        LOG.trace("Dispatching event message ({}) for topic ({}).", (Object)eventMessage, (Object)this);
        Collection<IEventDeserializer<Object>> deserializers = this.getDeserializers(eventMessage);
        if (deserializers.isEmpty()) {
            LOG.debug("No deserializers found in topic ({}) for event type ({}).", (Object)this, (Object)eventMessage.getType());
            return;
        }
        Preconditions.checkState((boolean)eventMessage.getPayload().hasArray());
        for (IEventDeserializer<Object> deserializer : deserializers) {
            Object event;
            try {
                event = deserializer.deserializeEvent(eventMessage.getPayload().array());
                LOG.trace("Deserialized event message ({}) using ({}) to ({}).", new Object[]{eventMessage, deserializer, event});
            }
            catch (Exception | LinkageError e) {
                LOG.error("Unable to deserialized event message ({}, topic {}) using ({}). {}", new Object[]{eventMessage, this.getId(), deserializer, ExceptionUtils.getRootCause((Throwable)e), e});
                continue;
            }
            for (Class<?> eventType : this.getReflectionService().getHierarchy(event.getClass())) {
                for (EventHandler handler : this.getHandlers(eventType)) {
                    this.dispatchEvent(event, handler);
                }
            }
        }
    }

    @VisibleForTesting
    void dispatchEvent(Object event, EventHandler handler) {
        LOG.trace("Dispatching event ({}) to handler ({})", event, (Object)handler);
        try {
            handler.handleEvent(event);
        }
        catch (Exception | LinkageError e) {
            LOG.error("Unable to dispatch event ({}, topic {}) to handler ({}). {}", new Object[]{event, this.getId(), handler, ExceptionUtils.getRootCause((Throwable)e), e});
        }
    }

    @VisibleForTesting
    Collection<IEventDeserializer<Object>> getDeserializers(EventMessage eventMessage) {
        return this.deserializersByEventTypeClassName.get((Object)eventMessage.getType());
    }

    EventService getEventService() {
        return this.eventService;
    }

    @VisibleForTesting
    List<EventHandler> getHandlers(Class<?> eventType) {
        this.eventHandlersByTypeLock.readLock().lock();
        try {
            ArrayList arrayList = Lists.newArrayList((Iterable)this.eventHandlersByType.get(eventType));
            return arrayList;
        }
        finally {
            this.eventHandlersByTypeLock.readLock().unlock();
        }
    }

    @Override
    public String getId() {
        return this.id;
    }

    ReflectionService getReflectionService() {
        return this.getEventService().getReflectionService();
    }

    @VisibleForTesting
    IEventSerializer<Object> getSerializer(Class<?> eventType) {
        IEventSerializer serializer = (IEventSerializer)this.serializersByEventType.get(eventType);
        if (serializer != null) {
            LOG.trace("Found serializer ({}) for event type ({})", (Object)serializer, eventType);
            return serializer;
        }
        for (Class<?> type : this.getReflectionService().getHierarchy(eventType)) {
            serializer = (IEventSerializer)this.serializersByEventType.get(type);
            if (serializer == null) continue;
            this.serializersByEventType.putIfAbsent(type, serializer);
            LOG.trace("Found serializer ({}) registered for type ({}) for event type ({})", new Object[]{serializer, type, eventType});
            return serializer;
        }
        throw new IllegalArgumentException(String.format("no serializer found for event of type '%s'", eventType));
    }

    @Override
    public void register(Object object) throws IllegalArgumentException {
        LOG.debug("Registering object ({}) with topic ({})", object, (Object)this);
        for (EventHandler handler : this.getReflectionService().getEventHandlers(object)) {
            this.registerHandler(handler);
        }
    }

    @VisibleForTesting
    void registerHandler(EventHandler handler) {
        LOG.debug("Registering handler ({}) with topic ({})", (Object)handler, (Object)this);
        this.eventHandlersByTypeLock.writeLock().lock();
        try {
            this.checkClosed();
            this.eventHandlersByType.put(handler.getEventType(), (Object)handler);
        }
        finally {
            this.eventHandlersByTypeLock.writeLock().unlock();
        }
        this.activateIfNecessary();
    }

    @Override
    public <T> void sendEvent(T event) throws IllegalArgumentException {
        LOG.trace("Sending event ({}) for topic ({})", event, (Object)this);
        this.checkClosed();
        byte[] serializedEvent = this.getSerializer(event.getClass()).serializeEvent(event);
        this.getEventService().queueEvent(this.getId(), this.createEventMessage(event.getClass(), serializedEvent));
    }

    @Override
    public void unregister(Object object) throws IllegalArgumentException {
        LOG.debug("Unregistering object ({}) from topic ({})", object, (Object)this);
        for (EventHandler handler : this.getReflectionService().getEventHandlers(object)) {
            this.unregisterHandler(handler);
        }
    }

    @VisibleForTesting
    void unregisterHandler(EventHandler handler) {
        LOG.debug("Unregistering handler ({}) from topic ({})", (Object)handler, (Object)this);
        this.eventHandlersByTypeLock.writeLock().lock();
        try {
            this.checkClosed();
            this.eventHandlersByType.remove(handler.getEventType(), (Object)handler);
        }
        finally {
            this.eventHandlersByTypeLock.writeLock().unlock();
        }
        this.deactivateIfPossible();
    }
}

