/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static org.eclipse.epp.internal.logging.aeri.ide.di.ImageRegistryCreationFunction.IMG_WIZBAN;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.SendMode;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;

public class SetupWizard extends Wizard {

    public static final String CTX_STATE_SETUP_IN_PROGRESS = BUNDLE_ID + ".di.configure-in-progress"; //$NON-NLS-1$

    private ISystemSettings systemSettings;
    private IEclipseContext parentContext;
    private List<IServerDescriptor> servers;

    private boolean onShellClosedEvent;

    @Inject
    public SetupWizard(ISystemSettings systemSettings, @Named(IDEWorkflow.CTX_SERVERS) List<IServerDescriptor> servers,
            ImageRegistry registry, IEclipseContext eclipseContext) {
        this.systemSettings = systemSettings;
        this.servers = servers;
        this.parentContext = eclipseContext;
        setHelpAvailable(false);
        setWindowTitle(Messages.WIZARD_TITLE_SETUP);
        setNeedsProgressMonitor(false);
        setDefaultPageImageDescriptor(registry.getDescriptor(IMG_WIZBAN));
        parentContext.modify(CTX_STATE_SETUP_IN_PROGRESS, true);
    }

    @Override
    public boolean performCancel() {
        if (onShellClosedEvent) {
            // do nothing
            return true;
        }

        for (IServerDescriptor server : servers) {
            server.setConfigured(true);
            server.setEnabled(false);
        }
        systemSettings.setSendMode(SendMode.NEVER);
        systemSettings.setConfigured(true);
        parentContext.modify(CTX_STATE_SETUP_IN_PROGRESS, false);
        return true;
    }

    @Override
    public boolean performFinish() {
        if (needsWarning()) {
            if (Window.OK != showAlert()) {
                return false;
            }
        }
        for (IServerDescriptor server : servers) {
            server.setConfigured(true);
        }
        systemSettings.setConfigured(true);
        parentContext.modify(CTX_STATE_SETUP_IN_PROGRESS, false);
        return true;
    }

    private int showAlert() {
        return new ExternalServersEnabledAlert(getShell()).open();
    }

    private boolean needsWarning() {
        for (IServerDescriptor server : servers) {
            String id = server.getId();
            if (server.isEnabled() && !id.startsWith("org.eclipse")) { //$NON-NLS-1$
                return true;
            }
        }
        return false;
    }

    public void notifyShellClosedEvent() {
        onShellClosedEvent = true;

    }
}
