/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static com.google.common.base.Optional.*;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.BUNDLE_ID;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.logging.aeri.core.ProblemStatus;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.DecorationOverlayIcon;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import com.google.common.base.Optional;

public class UIUtils {
    private static final Point TOOLTIP_DISPLACEMENT = new Point(5, 20);
    private static final int TOOLTIP_HIDE_DELAY = (int) SECONDS.toMillis(20);
    private static final String NEEDINFO_DECORATION_IMAGE = ISharedImages.IMG_DEC_FIELD_WARNING;

    public static Image decorate(@Nullable IServerDescriptor server, @Nullable ProblemStatus status, @Nullable ImageRegistry registry) {
        if (isAnyNull(server, status, registry)) {
            return ImageDescriptor.getMissingImageDescriptor().createImage();
        }
        String decoImageId = null;
        switch (status) {
        case NEW:
            decoImageId = "icons/ovr16/bllt-grn-pls.png"; //$NON-NLS-1$
            break;
        case UNCONFIRMED:
            decoImageId = "icons/ovr16/qstn-mrk.png"; //$NON-NLS-1$
            break;
        case CONFIRMED:
            decoImageId = "icons/ovr16/tck-bl.png"; //$NON-NLS-1$
            break;
        case NEEDINFO:
            decoImageId = "icons/ovr16/needinfo.png"; //$NON-NLS-1$
            break;
        case FIXED:
            decoImageId = "icons/ovr16/tck-grn.png"; //$NON-NLS-1$
            break;
        case IGNORED:
            decoImageId = "icons/ovr16/tck-grn.png"; //$NON-NLS-1$
            break;
        case WONTFIX:
        case INVALID:
            decoImageId = "icons/ovr16/bllt-gr.png"; //$NON-NLS-1$
            break;
        case FAILURE:
            decoImageId = "icons/ovr16/xclmtn-mrk.png"; //$NON-NLS-1$
            break;
        }
        String imageId = server.getId() + '-' + decoImageId;
        Image image = registry.get(imageId);
        if (image == null) {
            Image baseImage = server.getImage16();
            ImageDescriptor overlayDesc = AbstractUIPlugin.imageDescriptorFromPlugin(BUNDLE_ID, decoImageId);
            DecorationOverlayIcon decoratedImage = new DecorationOverlayIcon(baseImage,
                    new ImageDescriptor[] { null, overlayDesc, null, null, null }, new Point(16, 16));
            registry.put(imageId, decoratedImage);
            image = registry.get(imageId);

        }
        return checkNotNull(image);
    }

    private static boolean isAnyNull(Object... arguments) {
        for (Object argument : arguments) {
            if (argument == null) {
                return true;
            }
        }
        return false;
    }

    public static Image decorate(@Nullable IServerDescriptor server, @Nullable IStatus status, @Nullable ImageRegistry registry) {
        return decorate(server, status, registry, false);
    }

    public static Image decorate(@Nullable IServerDescriptor server, @Nullable IStatus status, @Nullable ImageRegistry registry,
            boolean needinfo) {
        if (isAnyNull(server, status, registry)) {
            return ImageDescriptor.getMissingImageDescriptor().createImage();
        }
        Image baseImage = server.getImage16();
        String imageId = server.getId();
        String imageIdNeedinfo = imageId + "-needinfo"; // $NON-NLS-1$
        String id = needinfo ? imageIdNeedinfo : imageId;
        Image image = registry.get(id);
        if (image == null) {
            registry.put(imageId, baseImage);
            DecorationOverlayIcon decoratedImageNeedinfo = decorateNeedinfo(baseImage);
            registry.put(imageIdNeedinfo, decoratedImageNeedinfo);
            image = registry.get(id);
        }
        return image;

    }

    private static DecorationOverlayIcon decorateNeedinfo(Image baseImage) {
        ImageDescriptor overlayNeedinfo = AbstractUIPlugin.imageDescriptorFromPlugin(BUNDLE_ID, "icons/ovr16/needinfo.png");
        DecorationOverlayIcon decoratedImageNeedinfo = new DecorationOverlayIcon(baseImage,
                new ImageDescriptor[] { overlayNeedinfo, null, null, null, null }, new Point(16, 16));
        return decoratedImageNeedinfo;
    }

    public static Image decorate(@Nullable IProcessorDescriptor processorDescriptor, @Nullable ImageRegistry registry, boolean needinfo) {
        if (isAnyNull(processorDescriptor, registry)) {
            return ImageDescriptor.getMissingImageDescriptor().createImage();
        }

        Image baseImage = processorDescriptor.getImage16();

        String imageId = processorDescriptor.getDirective() + "-16x16-"; //$NON-NLS-1$
        String imageIdNeedinfo = imageId + "-needinfo";
        String id = needinfo ? imageIdNeedinfo : imageId;
        Image image = registry.get(id);
        if (image == null) {
            registry.put(imageId, baseImage);

            DecorationOverlayIcon decoratedImageNeedinfo = decorateNeedinfo(registry.get(imageId));
            registry.put(imageIdNeedinfo, decoratedImageNeedinfo);
            image = registry.get(id);
        }
        return image;
    }

    public static Optional<Shell> getWorkbenchWindowShell() {
        IWorkbenchWindow workbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        if (workbenchWindow != null) {
            Shell shell = workbenchWindow.getShell();
            if (shell != null) {
                return Optional.of(shell);
            }
        }
        return Optional.absent();
    }

    public static boolean isUIThread() {
        return Display.getCurrent() != null;
    }

    /**
     * Returns the Display of <code>PlatformUI.getWorkbench().getDisplay()</code>. If the workbench is closing or not created or the display
     * is disposed this will return {@link Optional#absent()}.
     */
    public static Optional<Display> getDisplay() {
        IWorkbench workbench = PlatformUI.getWorkbench();
        if (workbench == null || workbench.isClosing()) {
            return absent();
        }
        Display display = workbench.getDisplay();
        if (display == null || display.isDisposed()) {
            return absent();
        }
        return of(display);
    }

}
