/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.filters;

import static org.eclipse.epp.logging.aeri.core.filters.AcceptedPackagesFilter.TestResult.UNDEF;

import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.logging.aeri.core.util.StatusSwitch;
import org.eclipse.jdt.annotation.Nullable;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;

public class AcceptedPackagesFilter implements Predicate<IStatus> {

    private List<Pattern> patterns;
    private boolean strict;

    /**
     * @param patterns
     *            the list of package patterns to match
     */
    public AcceptedPackagesFilter(List<Pattern> patterns) {
        this(patterns, true);
    }

    /**
     *
     * @param patterns
     *            the list of package patterns to match
     * @param strict
     *            whether this filter should accept statuses that contain <b>also</b> other packages than those specified in patterns. Note
     *            that at least one package has to be matched.
     */
    public AcceptedPackagesFilter(List<Pattern> patterns, boolean strict) {
        this.patterns = patterns;
        this.strict = strict;
    }

    @Override
    public boolean apply(IStatus input) {
        if (strict) {
            return applyStrictMatching(input);
        } else {
            return applyNonStrictMatching(input);
        }
    }

    private boolean applyNonStrictMatching(IStatus input) {
        NonStrictStatusSwitch test = new NonStrictStatusSwitch();
        TestResult res = Objects.firstNonNull(test.doSwitch(input), UNDEF);
        switch (res) {
        case PASS:
            return true;
        case FAIL:
            // At the time of writing, this will actually never be returned but...
            return false;
        case UNDEF:
        default:
            return test.packageMatch;
        }
    }

    private boolean applyStrictMatching(IStatus input) {
        TestResult res = Objects.firstNonNull(new StrictStatusSwitch().doSwitch(input), UNDEF);
        switch (res) {
        case FAIL:
            return false;
        case PASS:
        case UNDEF:
        default:
            return true;
        }
    }

    private final class StrictStatusSwitch extends StatusSwitch<TestResult> {
        @Override
        @Nullable
        public TestResult caseStackTraceElement(StackTraceElement stackTraceElement) {
            String label = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
            for (Pattern pattern : patterns) {
                if (!pattern.matcher(label).matches()) {
                    return TestResult.FAIL;
                }
            }
            return null;
        }
    }

    private final class NonStrictStatusSwitch extends StatusSwitch<TestResult> {
        boolean packageMatch = false;

        @Override
        @Nullable
        public TestResult caseStackTraceElement(StackTraceElement stackTraceElement) {
            String label = stackTraceElement.getClassName() + "." + stackTraceElement.getMethodName();
            for (Pattern pattern : patterns) {
                if (!pattern.matcher(label).matches()) {
                    // We found an 'unexpected' package
                    // Did we find (at least) one expected package before?
                    if (packageMatch) {
                        // if so, stop early and accept this status;
                        return TestResult.PASS;
                    }
                    return null;
                } else {
                    packageMatch = true;
                }
            }
            return null;
        }
    }

    enum TestResult {
        PASS, FAIL, UNDEF
    }
}
