/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.notifications;

import static org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow.TOPIC_USER_REQUESTS_SEND_ONE_GROUP;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.IDEConstants.NOTIFY_REPORT;

import java.util.List;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.e4.core.contexts.ContextInjectionFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.core.services.events.IEventBroker;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEvent;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEventGroup;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.dialogs.UIUtils;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.OpenReviewDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.preference.JFacePreferences;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.mylyn.commons.ui.compatibility.CommonColors;
import org.eclipse.mylyn.commons.workbench.forms.ScalingHyperlink;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;

@SuppressWarnings("restriction")
public class NewEventNotification extends Notification {

    private ILogEventGroup group;
    private ImageRegistry registry;
    private IEclipseContext context;
    private OpenReviewDialogHandler handler;

    @Inject
    public NewEventNotification(ILogEventGroup group, ImageRegistry registry, IEventBroker broker, IEclipseContext context,
            OpenReviewDialogHandler handler) {

        super(NOTIFY_REPORT, broker);
        this.group = group;
        this.registry = registry;
        this.context = context;
        this.handler = handler;
    }

    public Control createControl(Composite parent) {
        GridDataFactory asis = GridDataFactory.fillDefaults().align(SWT.FILL, SWT.TOP);
        GridDataFactory w400 = GridDataFactory.fillDefaults().hint(NotificationPopup.MAX_WIDTH, SWT.DEFAULT).grab(false, false)
                .align(SWT.FILL, SWT.TOP);
        GridDataFactory w360 = GridDataFactory.fillDefaults().hint(NotificationPopup.MAX_WIDTH - 30, SWT.DEFAULT).grab(false, false)
                .align(SWT.FILL, SWT.TOP);
        Composite container = new Composite(parent, SWT.NO_FOCUS);
        GridLayoutFactory.fillDefaults().extendedMargins(0, 0, 0, 0).applyTo(container);
        w400.applyTo(container);

        final Link label = new Link(container, SWT.WRAP | SWT.NO_FOCUS);
        label.setText(getLabel());
        label.setForeground(CommonColors.TEXT_QUOTED);
        w400.applyTo(label);

        final Link description = new Link(container, SWT.WRAP | SWT.NO_FOCUS);
        description.setText(getDescription());
        w400.applyTo(description);

        Composite serverSections = new Composite(container, SWT.NO_FOCUS);
        GridLayoutFactory.fillDefaults().numColumns(2).equalWidth(false).applyTo(serverSections);

        for (ILogEvent event : group.getEvents()) {
            IServerDescriptor server = event.getServer();
            // skip sections for not yet configured servers - TODO shouldn't this be done elsewhere?
            if (!server.isActive()) {
                continue;
            }
            final IProblemState interest = event.getInterest();
            Label icon = new Label(serverSections, SWT.NO_FOCUS);
            Image image = UIUtils.decorate(server, interest.getStatus(), registry);
            icon.setImage(image);
            asis.applyTo(icon);

            Link response = new Link(serverSections, SWT.NO_FOCUS | SWT.WRAP);
            w360.applyTo(response);
            response.setText(getMessage(server, interest));
            response.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    Browsers.openInExternalBrowser(e.text);
                }
            });
        }
        {
            final Link label2 = new Link(container, SWT.WRAP | SWT.NO_FOCUS);
            label2.setText(Messages.LINK_TEXT_DO_YOU_WANT_TO_REPORT_ERROR);
            w400.applyTo(label);
        }
        createLinkSection(container);
        return container;
    }

    private String getMessage(IServerDescriptor server, final IProblemState interest) {
        String message = interest.getMessage();
        if (!StringUtils.isBlank(message)) {
            return message;
        }
        switch (interest.getStatus()) {
        case NEW:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_NEW_PROBLEM, server.getName());
        case UNCONFIRMED:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_UNCONFIRMED_PROBLEM, server.getName());
        case CONFIRMED:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_CONFIRMED_PROBLEM, server.getName());
        case NEEDINFO:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_NEEDINFO_PROBLEM, server.getName());
        case FIXED:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_FIXED_PROBLEM, server.getName());
        case IGNORED:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_IGNORED_PROBLEM, server.getName());
        case INVALID:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_INVALID_PROBLEM, server.getName());
        case FAILURE:
            return Formats.format(Messages.LINK_TEXT_DEFAULT_MESSAGE_FAILURE, server.getName());
        default:
            return ""; //$NON-NLS-1$
        }
    }

    protected void createLinkSection(Composite container) {
        Color linkColor = JFaceResources.getColorRegistry().get(JFacePreferences.HYPERLINK_COLOR);
        // Links;
        Composite linksComposite = new Composite(container, SWT.NO_FOCUS);
        GridDataFactory.fillDefaults().align(SWT.END, SWT.TOP).applyTo(linksComposite);
        GridLayoutFactory.fillDefaults().numColumns(3).margins(40, 0).applyTo(linksComposite);

        // separator one
        GridDataFactory.fillDefaults().grab(true, false).applyTo(new Label(linksComposite, SWT.NONE));
        {
            final ScalingHyperlink showLink = new ScalingHyperlink(linksComposite, SWT.NO_FOCUS);
            GridDataFactory.swtDefaults().applyTo(showLink);
            showLink.setText(Messages.LINK_TEXT_VIEW_DETAILS);
            showLink.setForeground(linkColor);
            showLink.registerMouseTrackListener();
            showLink.addHyperlinkListener(new HyperlinkAdapter() {
                @Override
                public void linkActivated(HyperlinkEvent e) {
                    closeWithEvent(null, null);
                    ContextInjectionFactory.invoke(handler, Execute.class, context);
                }
            });
        }
        {
            final ScalingHyperlink sendLink = new ScalingHyperlink(linksComposite, SWT.NO_FOCUS);
            GridDataFactory.swtDefaults().applyTo(sendLink);
            sendLink.setText(Messages.LINK_TEXT_SEND);
            sendLink.setForeground(linkColor);
            sendLink.registerMouseTrackListener();
            sendLink.addHyperlinkListener(new HyperlinkAdapter() {

                @Override
                public void linkActivated(HyperlinkEvent e) {
                    closeWithEvent(TOPIC_USER_REQUESTS_SEND_ONE_GROUP, group);
                }
            });
        }
    }

    protected String getResponse(IServerDescriptor server, IProblemState interest) {
        switch (interest.getStatus()) {
        case NEW:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_NEW, server.getName(), interest.getMessage());
        case UNCONFIRMED:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_UNCONFIRMED, server.getName(), interest.getMessage());
        case CONFIRMED:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_CONFIRMED, server.getName(), interest.getMessage());
        case NEEDINFO:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_NEEDINFO, server.getName(), interest.getMessage());
        case FIXED:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_FIXED, server.getName(), interest.getMessage());
        case IGNORED:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_IGNORED, server.getName(), interest.getMessage());
        case INVALID:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_INVALID, server.getName(), interest.getMessage());
        case FAILURE:
            return Formats.format(Messages.NOTIFY_INTEREST_PROBLEM_FAILURE, server.getName(), interest.getMessage());
        default:
            return ""; //$NON-NLS-1$
        }
    }

    @Override
    public String getTitle() {
        return Formats.format(Messages.NOTIFICATION_TITLE_NEW_EVENT, group.getStatus().getPlugin());
    }

    @Override
    public String getLabel() {
        return Formats.format(Messages.NOTIFICATION_LABEL_NEW_EVENT, StringUtils.abbreviate(group.getStatus().getMessage(), 80));
    }

    @Override
    public String getDescription() {
        Throwable exception = Preconditions.checkNotNull(group.getStatus().getException());
        String exceptionMessage = Optional.fromNullable(exception.getMessage()).or(""); //$NON-NLS-1$
        String exceptionHeadline = StringUtils.abbreviate(
                Formats.format(Messages.NOTIFICATION_DESCRIPTION_NEW_EVENT, exception.getClass().getSimpleName(), exceptionMessage), 80);
        return exceptionHeadline;
    }

    @Override
    public List<NotificationAction> getActions() {
        throw new IllegalStateException("Operation not supported"); //$NON-NLS-1$
    }
}
