/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.server.mars;

import static java.util.concurrent.TimeUnit.*;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_INVALID_PATTERN;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import org.eclipse.epp.logging.aeri.core.filters.StatusIgnorePattern;
import org.eclipse.epp.logging.aeri.core.util.WildcardPatterns;
import org.eclipse.jdt.annotation.Nullable;

import com.google.common.collect.Lists;

public class ServerConfiguration {

    private String version;
    private String title;
    private String description;
    private long timestamp; // when did we discover the service the last time?
    private long ttl; // (in minutes) how long is this service configuration valid?
    private String helpUrl;
    private String feedbackUrl;
    private String aboutUrl;
    private String submitUrl;
    private int maxReportSize;
    private String problemsUrl;

    // in minutes
    private long problemsTtl;
    private String interestUrl;

    // max time in seconds until a connection to the server has to be established.
    private int connectTimeout = (int) TimeUnit.SECONDS.toMillis(3);;
    // max time in seconds between two packets sent back to the client.
    private int socketTimeout = (int) TimeUnit.SECONDS.toMillis(10);;

    private List<String> acceptedProducts;
    @Nullable
    private transient List<Pattern> acceptedProductsPatterns;
    private List<String> acceptedPlugins;
    @Nullable
    private transient List<Pattern> acceptedPluginsPatterns;
    private List<String> acceptedPackages;
    @Nullable
    private transient List<Pattern> acceptedPackagesPatterns;

    private List<String> requiredPackages;
    @Nullable
    private transient List<Pattern> requiredPackagesPatterns;

    /**
     * Whether or not stacktraces with other than the accepted packages will be send (=true) or discarded (=false)
     */
    private boolean acceptOtherPackages;
    private boolean acceptUiFreezes;

    /**
     * Patterns to ignore errors</br>
     * format: <code>pluginId:exception:message</code>. use '*' for wildcard or leave empty (e.g. '
     * <code>::<code>' is the minimal pattern to ignore everything )
     * <p>
     * prefix for pluginId: </br>
     * '^' : check only the first status </br>
     * none : check every status and it's children
     * <p>
     * prefix for exception: </br>
     * '^' : only top of stacktrace </br>
     * '$' : only last of stacktrace </br>
     * none : anywhere in stacktrace
     * <p>
     * example: <code>org.eclipse.epp.logging.aeri.*:^*StandInException:*</code> </br>
     * to ignore all exceptions from plugins starting with <code>org.eclipse.epp.logging.aeri.</code> on top of a stacktrace, ending with
     * <code>StandInException</code> with any message. The status may be included in other status-objects.
     */
    private List<String> ignoredStatuses;
    @Nullable
    private transient List<StatusIgnorePattern> ignoredPatterns;
    private long problemsZipLastDownloadTimestamp;

    public int getConnectTimeoutMs() {
        return (int) MILLISECONDS.convert(connectTimeout, SECONDS);
    }

    public int getConnectTimeout() {
        return connectTimeout;
    }

    public void setConnectTimeout(int connectTimeout) {
        this.connectTimeout = connectTimeout;
    }

    public int getSocketTimeoutMs() {
        return (int) MILLISECONDS.convert(socketTimeout, SECONDS);
    }

    public long getSocketTimeout() {
        return socketTimeout;
    }

    public void setSocketTimeout(int socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Returns the time-to-live of these settings in minutes, i.e., informs the client when he should refresh the settings.
     */
    public long getTtl() {
        return ttl;
    }

    public long getTtlMs() {
        return MILLISECONDS.convert(getTtl(), MINUTES);
    }

    public void setTtl(long ttlInMinutes) {
        this.ttl = ttlInMinutes;
    }

    public String getHelpUrl() {
        return helpUrl;
    }

    public void setHelpUrl(String helpUrl) {
        this.helpUrl = helpUrl;
    }

    public String getFeedbackUrl() {
        return feedbackUrl;
    }

    public void setFeedbackUrl(String feedbackUrl) {
        this.feedbackUrl = feedbackUrl;
    }

    public String getAboutUrl() {
        return aboutUrl;
    }

    public void setAboutUrl(String aboutUrl) {
        this.aboutUrl = aboutUrl;
    }

    public String getSubmitUrl() {
        return submitUrl;
    }

    public void setSubmitUrl(String submitUrl) {
        this.submitUrl = submitUrl;
    }

    public int getMaxReportSize() {
        return maxReportSize;
    }

    public void setMaxReportSize(int submitSizeLimit) {
        this.maxReportSize = submitSizeLimit;
    }

    public String getProblemsUrl() {
        return problemsUrl;
    }

    public void setProblemsUrl(String problemsUrl) {
        this.problemsUrl = problemsUrl;
    }

    /**
     * Returns the time-to-live of the error reports database in minutes, i.e., informs the client when he should refresh the database.
     */
    public long getProblemsTtl() {
        return problemsTtl;
    }

    public long getProblemsTtlMs() {
        return MILLISECONDS.convert(getProblemsTtl(), MINUTES);
    }

    public void setProblemsTtl(long problemsTtlInMinutes) {
        this.problemsTtl = problemsTtlInMinutes;
    }

    public String getInterestUrl() {
        return interestUrl;
    }

    public void setInterestUrl(String interestUrl) {
        this.interestUrl = interestUrl;
    }

    public List<String> getAcceptedProducts() {
        return acceptedProducts;
    }

    public void setAcceptedProducts(List<String> acceptedProducts) {
        this.acceptedProducts = acceptedProducts;
        this.acceptedProductsPatterns = null;
    }

    public List<String> getAcceptedPlugins() {
        return acceptedPlugins;
    }

    public void setAcceptedPlugins(List<String> acceptedPlugins) {
        this.acceptedPlugins = acceptedPlugins;
        this.acceptedPluginsPatterns = null;
    }

    public List<Pattern> getAcceptedProductsPatterns() {
        if (acceptedProductsPatterns == null) {
            acceptedProductsPatterns = WildcardPatterns.convert(acceptedProducts);
        }
        return acceptedProductsPatterns;
    }

    public List<Pattern> getAcceptedPluginsPatterns() {
        if (acceptedPluginsPatterns == null) {
            acceptedPluginsPatterns = WildcardPatterns.convert(acceptedPlugins);
        }
        return acceptedPluginsPatterns;
    }

    public List<Pattern> getAcceptedPackagesPatterns() {
        if (acceptedPackagesPatterns == null) {
            acceptedPackagesPatterns = WildcardPatterns.convert(acceptedPackages);
        }
        return acceptedPackagesPatterns;
    }

    public List<String> getAcceptedPackages() {
        return acceptedPackages;
    }

    public void setAcceptedPackages(List<String> acceptedPackages) {
        this.acceptedPackages = acceptedPackages;
        this.acceptedPackagesPatterns = null;
    }

    public List<Pattern> getRequiredPackagesPatterns() {
        if (requiredPackagesPatterns == null) {
            requiredPackagesPatterns = WildcardPatterns.convert(requiredPackages);
        }
        return requiredPackagesPatterns;
    }

    public List<String> getRequiredPackages() {
        return requiredPackages;
    }

    public void setRequiredPackages(List<String> requiredPackages) {
        this.requiredPackages = requiredPackages;
        this.requiredPackagesPatterns = null;
    }

    public boolean isAcceptOtherPackages() {
        return acceptOtherPackages;
    }

    public void setAcceptOtherPackages(boolean acceptOtherPackages) {
        this.acceptOtherPackages = acceptOtherPackages;
    }

    public List<StatusIgnorePattern> getIgnoredPluginMessagesPatterns() {
        if (ignoredPatterns == null) {
            ignoredPatterns = Lists.newArrayList();
            for (String s : getIgnoredStatuses()) {
                StatusIgnorePattern pattern = StatusIgnorePattern.fromString(s);
                if (pattern != null) {
                    ignoredPatterns.add(pattern);
                } else {
                    log(WARN_INVALID_PATTERN, s);
                }
            }
        }
        return ignoredPatterns;
    }

    public boolean isAcceptUiFreezes() {
        return acceptUiFreezes;
    }

    public void setAcceptUiFreezes(boolean acceptUiFreezes) {
        this.acceptUiFreezes = acceptUiFreezes;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public long getProblemsZipLastDownloadTimestamp() {
        return problemsZipLastDownloadTimestamp;
    }

    public void setProblemsZipLastDownloadTimestamp(long problemsZipLastDownloadTimestamp) {
        this.problemsZipLastDownloadTimestamp = problemsZipLastDownloadTimestamp;
    }

    public List<String> getIgnoredStatuses() {
        return ignoredStatuses;
    }

    public void setIgnoredStatuses(List<String> ignoredStatuses) {
        this.ignoredStatuses = ignoredStatuses;
    }
}
