/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.dialogs;

import static java.lang.Math.max;
import static org.eclipse.emf.databinding.EMFProperties.value;
import static org.eclipse.epp.internal.logging.aeri.ide.IIdePackage.Literals.SERVER_DESCRIPTOR__ENABLED;
import static org.eclipse.epp.internal.logging.aeri.ide.dialogs.UI.*;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Formats.format;
import static org.eclipse.epp.logging.aeri.core.SystemControl.executeHandler;
import static org.eclipse.epp.logging.aeri.core.util.Links.REL_PROVIDER;
import static org.eclipse.jface.databinding.swt.WidgetProperties.selection;

import java.lang.reflect.Method;
import java.util.Collection;

import javax.inject.Inject;
import javax.inject.Named;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.conversion.Converter;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.ui.di.UISynchronize;
import org.eclipse.epp.internal.logging.aeri.ide.IDEWorkflow;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.handlers.CreateConfigureDialogHandler;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Browsers;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.logging.aeri.core.ILink;
import org.eclipse.epp.logging.aeri.core.util.Links;
import org.eclipse.jface.databinding.swt.WidgetProperties;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.Hyperlink;

import com.google.common.base.Objects;

public class ServersPage extends WizardPage {

    private Collection<IServerDescriptor> servers;
    private IEclipseContext parentContext;
    private DataBindingContext context;
    private UISynchronize uiSynchronize;

    @Inject
    public ServersPage(@Named(IDEWorkflow.CTX_SERVERS) Collection<IServerDescriptor> servers, UISynchronize uiSynchronize,
            IEclipseContext parentContext) {
        super(ServersPage.class.getName());
        this.servers = servers;
        this.uiSynchronize = uiSynchronize;
        this.parentContext = parentContext;
        context = new DataBindingContext();
        setPageComplete(false);

        setTitle(format(Messages.WIZPAGE_TITLE_SERVERS, servers.size()));
        setDescription(Messages.WIZPAGE_DESCRIPTION_SERVERS);
    }

    @Override
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
            setPageComplete(true);
            try {
                Method method = Dialog.class.getDeclaredMethod("getButton", int.class); //$NON-NLS-1$
                method.setAccessible(true);
                Button button = (Button) method.invoke(getContainer(), IDialogConstants.FINISH_ID);
                button.setFocus();
            } catch (Exception e) {
                // we tried
            }
        }
    }

    @Override
    public void createControl(Composite parent) {
        int preferredWidth = 400;
        Font font = JFaceResources.getDialogFont();
        int fontSize = font.getFontData()[0].getHeight();
        Font small = FontDescriptor.createFrom(font).setHeight(fontSize).createFont(parent.getDisplay());

        Composite container = new Composite(parent, SWT.NONE);
        // container.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_DARK_GREEN));

        gl().spacing(0, 3).extendedMargins(10, 0, 10, 0).applyTo(container);
        gdGrabHV().applyTo(container);
        {
            Label header = new Label(container, SWT.WRAP);
            header.setText(Messages.LIST_HEADER_INTERESTED_PROJECTS);
            gdGrabH().hint(preferredWidth, SWT.DEFAULT).applyTo(header);
        }

        ScrolledComposite scrolledComposite = new ScrolledComposite(container, SWT.H_SCROLL | SWT.V_SCROLL);
        scrolledComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
        gl().applyTo(scrolledComposite);
        scrolledComposite.setExpandVertical(true);
        scrolledComposite.setExpandHorizontal(true);

        final Composite sections = new Composite(scrolledComposite, SWT.NONE);
        gl().extendedMargins(3, 3, 3, 3).applyTo(sections);
        // gdGrabH().hint(preferredWidth, 300).applyTo(sections);
        gdGrabHV().hint(preferredWidth, 200).applyTo(scrolledComposite);
        scrolledComposite.setContent(sections);

        Font headlineFont = FontDescriptor.createFrom(font).setStyle(SWT.BOLD).createFont(parent.getDisplay());
        for (final IServerDescriptor server : servers) {
            Composite section = new Composite(sections, SWT.NONE);
            // section.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_DARK_RED));
            int sectionColumns = 5;
            gl().numColumns(sectionColumns).margins(0, 0).spacing(10, 1).applyTo(section);
            gdGrabH().applyTo(section);
            {
                Label icon = new Label(section, SWT.NONE);
                icon.setImage(server.getImage32());
                gd().indent(0, fontSize / 2).align(SWT.BEGINNING, SWT.BEGINNING).span(1, 3).applyTo(icon);
            }
            {
                Label headline = new Label(section, SWT.WRAP);
                headline.setText(server.getName());
                headline.setFont(headlineFont);
                gd().span(sectionColumns - 1, 1).applyTo(headline);
            }
            {
                Link description = new Link(section, SWT.WRAP);
                String text = StringUtils.defaultString(server.getDescription(), Messages.LINK_TEXT_MISSING_DESCRIPTION);
                description.setText(text);
                description.addSelectionListener(new SelectionAdapter() {
                    @Override
                    public void widgetSelected(SelectionEvent e) {
                        Browsers.openInExternalBrowser(e.text);
                    }
                });
                gdGrabH().span(sectionColumns - 1, 1).hint(preferredWidth - 10, SWT.DEFAULT).applyTo(description);
            }
            {
                ILink link = getProviderLink(server);
                Hyperlink provider = new Hyperlink(section, SWT.WRAP);
                provider.setText(Formats.format(Messages.LABEL_SERVER_PROVIDER, link.getTitle()));
                provider.setHref(link.getHref());
                provider.setToolTipText(link.getHref());
                provider.addHyperlinkListener(new HyperlinkAdapter() {

                    @Override
                    public void linkActivated(HyperlinkEvent e) {
                        Browsers.openInExternalBrowser(e.getHref());
                    }
                });
                provider.setFont(small);
                provider.setUnderlined(true);
                provider.setForeground(parent.getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY));
                gdGrabH().align(SWT.BEGINNING, SWT.BEGINNING).applyTo(provider);
            }
            {
                Link configure = new Link(section, SWT.WRAP);
                configure.setText(Messages.LINK_TEXT_CONFIGURE);
                configure.setFont(small);
                configure.addSelectionListener(new SelectionAdapter() {
                    @Override
                    public void widgetSelected(SelectionEvent e) {
                        final IEclipseContext child = parentContext.createChild();
                        child.set(IServerDescriptor.class, server);
                        uiSynchronize.asyncExec(new Runnable() {

                            @Override
                            public void run() {
                                Dialog page = (Dialog) executeHandler(CreateConfigureDialogHandler.class, child);
                                page.open();
                            }
                        });
                    }
                });
                gd().align(SWT.END, SWT.CENTER).applyTo(configure);
            }
            {
                final Button enable = new Button(section, SWT.CHECK | SWT.FLAT);
                enable.setFont(parent.getFont());

                BooleanToEnablementTextConverter textConverter = new BooleanToEnablementTextConverter();
                Point enabledSize = computeSizeForConvertedValue(enable, true, textConverter);
                Point disabledSize = computeSizeForConvertedValue(enable, false, textConverter);
                gd().align(SWT.END, SWT.CENTER).hint(max(enabledSize.x, disabledSize.x), max(enabledSize.y, disabledSize.y))
                        .applyTo(enable);

                IObservableValue swt = selection().observe(enable);
                IObservableValue emf = value(SERVER_DESCRIPTOR__ENABLED).observe(server);
                IObservableValue label = WidgetProperties.text().observe(enable);
                this.context.bindValue(swt, emf);
                this.context.bindValue(label, emf, null, new UpdateValueStrategy().setConverter(textConverter));
            }
        }
        {
            Link termsAndConditions = new Link(container, SWT.NONE);
            termsAndConditions.setText(Messages.LINK_TEXT_ENABLEMENT_ADMONITION);

            gdGrabH().indent(0, 10).align(SWT.FILL, SWT.END).hint(preferredWidth, SWT.DEFAULT).applyTo(termsAndConditions);
        }

        Point sectionsSize = sections.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sections.setSize(sectionsSize);
        scrolledComposite.setMinSize(sectionsSize);

        Dialog.applyDialogFont(container);

        setControl(container);
    }

    private Point computeSizeForConvertedValue(Button button, boolean value, BooleanToEnablementTextConverter textConverter) {
        button.setText((String) textConverter.convert(value));
        return button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
    }

    private ILink getProviderLink(final IServerDescriptor server) {
        ILink link = server.getLinks().get(REL_PROVIDER);
        if (link == null) {
            return Links.createProviderLink("http://invalid/", Messages.LINK_TOOLTIP_MISSING_URL); //$NON-NLS-1$
        } else {
            return link;
        }
    }

    @Override
    public void dispose() {
        context.dispose();
        super.dispose();
    }

    private final class BooleanToEnablementTextConverter extends Converter {

        private BooleanToEnablementTextConverter() {
            super(Boolean.class, String.class);
        }

        @Override
        public Object convert(Object fromObject) {
            Boolean value = Objects.firstNonNull((Boolean) fromObject, Boolean.FALSE);
            return value ? Messages.BUTTON_TEXT_ENABLED : Messages.BUTTON_TEXT_DISABLED;
        }
    }
}
