/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.server.rest;

import static org.apache.http.HttpStatus.SC_NOT_FOUND;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_REST_QUERY_FAILED;
import static org.eclipse.epp.internal.logging.aeri.ide.server.Proxies.*;
import static org.eclipse.epp.logging.aeri.core.util.Logs.log;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.cache.HttpCacheStorage;
import org.apache.http.client.fluent.Executor;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.utils.DateUtils;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.cache.CachingHttpClientBuilder;
import org.apache.http.impl.client.cache.CachingHttpClients;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages;
import org.eclipse.epp.internal.logging.aeri.ide.server.json.Json;
import org.eclipse.epp.internal.logging.aeri.ide.server.mars.IO;
import org.eclipse.epp.internal.logging.aeri.ide.server.mars.IProblemsHistory;
import org.eclipse.epp.internal.logging.aeri.ide.server.mars.ServerConfiguration;
import org.eclipse.epp.logging.aeri.core.ILink;
import org.eclipse.epp.logging.aeri.core.IModelFactory;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.epp.logging.aeri.core.ISystemSettings;
import org.eclipse.epp.logging.aeri.core.ProblemStatus;
import org.eclipse.epp.logging.aeri.core.util.Statuses;

import com.google.common.base.Optional;
import com.google.gson.reflect.TypeToken;

public class RestBasedProblemsHistory implements IProblemsHistory {

    private static final ContentType STATUS_REPONSES_CONTENT_TYPE = ContentType.create("application/x.aer.status-reponses+json");
    private static final String STACK_TRACE_FINGERPRINT__QUERY = "stackTraceFingerprint";

    private final ServerConfiguration config;
    private final URI baseURI;
    private final LuceneHttpCacheStorage storage;
    private final CloseableHttpClient client;
    private final Executor executor;

    private Date embargoDate = new Date();

    public RestBasedProblemsHistory(ServerConfiguration config, File cacheDir) throws IOException {
        this.config = config;
        try {
            this.baseURI = new URI(config.getInterestUrl());
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }

        Directory directory = FSDirectory.open(cacheDir);
        storage = new LuceneHttpCacheStorage(directory);
        client = createClient(storage);
        executor = Executor.newInstance(client);
    }

    @Override
    public Optional<IProblemState> seen(IStatus status) {
        Date now = new Date();
        if (now.before(embargoDate)) {
            log(LogMessages.INFO_TEMPORARILY_DISABLED_REST_QUERIES, config.getTitle());
            return Optional.absent();
        }

        String fingerprint = Statuses.traceIdentityHash(status);
        URI restURI;
        try {
            restURI = new URIBuilder(baseURI).addParameter(STACK_TRACE_FINGERPRINT__QUERY, fingerprint).build();
        } catch (URISyntaxException e1) {
            return Optional.absent();
        }

        try {
            Request request = Request.Get(restURI).addHeader(HttpHeaders.ACCEPT, STATUS_REPONSES_CONTENT_TYPE.toString())
                    .viaProxy(getProxyHost(restURI).orNull()).connectTimeout(config.getConnectTimeoutMs()).staleConnectionCheck(true)
                    .socketTimeout(config.getSocketTimeoutMs());
            StatusReponse statusReponse = proxyAuthentication(executor, restURI).execute(request).handleResponse(new RestReponseHandler());

            if (statusReponse != null) {
                return Optional.of(toProblemState(statusReponse));
            } else {
                return Optional.absent();
            }
        } catch (IOException e) {
            log(WARN_REST_QUERY_FAILED, e, config.getTitle(), restURI);
            return Optional.absent();
        }
    }

    @Override
    public void sync(IO io, ISystemSettings systemSettings) {
        // No-op
    }

    @Override
    public void close() throws IOException {
        client.close();
        storage.close();
    }

    /**
     * For legacy reasons, {@link IProblemState} is not a perfect match for {@link StatusReponse}, but this hopefully will change in the
     * future.
     */
    private IProblemState toProblemState(StatusReponse serverResponse) {
        IProblemState mProblemState = IModelFactory.eINSTANCE.createProblemState();
        mProblemState.setStatus(toProblemStatus(serverResponse.getSituation()));
        mProblemState.setMessage(serverResponse.getMessage());
        for (String auxiliaryInformationRequest : serverResponse.getAuxiliaryInformationRequests()) {
            mProblemState.getNeedinfo().add(auxiliaryInformationRequest);
        }
        for (Link link : serverResponse.getLinks()) {
            ILink mLink = IModelFactory.eINSTANCE.createLink();
            mLink.setHref(link.getHref().toString());
            mLink.setRel(link.getRel());
            mLink.setTitle(link.getTitle());
            mProblemState.getLinks().put(link.getRel(), mLink);
        }
        return mProblemState;
    }

    private ProblemStatus toProblemStatus(ProblemSituation situation) {
        switch (situation) {
        case FAILURE:
            return ProblemStatus.FAILURE;
        case FIXED:
            return ProblemStatus.FIXED;
        case IGNORE:
            return ProblemStatus.IGNORED;
        case OPEN:
            return ProblemStatus.CONFIRMED;
        case WONTFIX:
            return ProblemStatus.WONTFIX;
        default:
            throw new IllegalArgumentException(situation.toString());
        }
    }

    private final class RestReponseHandler implements ResponseHandler<StatusReponse> {
        @Override
        public StatusReponse handleResponse(HttpResponse response) throws IOException {
            StatusLine statusLine = response.getStatusLine();
            HttpEntity entity = response.getEntity();
            if (statusLine.getStatusCode() == SC_NOT_FOUND) {
                return null;
            } else if (statusLine.getStatusCode() >= HttpStatus.SC_INTERNAL_SERVER_ERROR) {
                Header retryAfterHeader = response.getFirstHeader(HttpHeaders.RETRY_AFTER);
                if (retryAfterHeader != null) {
                    embargoDate = parseRetryAfter(retryAfterHeader.getValue());
                }
                throw new HttpResponseException(statusLine.getStatusCode(), statusLine.getReasonPhrase());
            } else if (statusLine.getStatusCode() >= HttpStatus.SC_MULTIPLE_CHOICES) {
                throw new HttpResponseException(statusLine.getStatusCode(), statusLine.getReasonPhrase());
            }

            if (entity == null) {
                throw new ClientProtocolException("Response contains no content");
            }
            ContentType contentType = ContentType.getOrDefault(entity);
            // Ignore any parameters sent by the server, as application/json doesn't have any as per RFC 7159
            if (!contentType.getMimeType().equals(STATUS_REPONSES_CONTENT_TYPE.getMimeType())) {
                throw new ClientProtocolException("Unexpected content type: " + contentType);
            }

            try {
                List<StatusReponse> statusResponses = Json.deserialize(entity.getContent(), new TypeToken<List<StatusReponse>>() {
                }.getType());
                if (statusResponses.isEmpty()) {
                    throw new IOException("Expected at least one status response");
                }
                return statusResponses.get(0);
            } catch (Exception e) {
                throw new ClientProtocolException("Cannot parse content", e);
            }
        }
    }

    /**
     * Workaround for <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=508270">Bug 508270</a>
     */
    private CloseableHttpClient createClient(HttpCacheStorage storage) {
        CachingHttpClientBuilder builder = CachingHttpClients.custom().setHttpCacheStorage(storage);
        try {
            Method buildMethod = CachingHttpClientBuilder.class.getMethod("build");
            return (CloseableHttpClient) buildMethod.invoke(builder);
        } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException
                | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    private Date parseRetryAfter(String retryAfter) {
        if (retryAfter != null) {
            try {
                return new Date(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(Long.parseLong(retryAfter)));
            } catch (NumberFormatException ignore) {
                Date date = DateUtils.parseDate(retryAfter);
                return date != null ? date : tomorrow();
            }
        } else {
            return tomorrow();
        }
    }

    private Date tomorrow() {
        return new Date(System.currentTimeMillis() + TimeUnit.DAYS.toMillis(1));
    }
}
