/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static org.apache.commons.io.filefilter.DirectoryFileFilter.DIRECTORY;
import static org.apache.commons.io.filefilter.FileFileFilter.FILE;
import static org.apache.commons.lang3.StringUtils.removeStart;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.apache.commons.io.FileUtils;

import com.google.common.io.Closeables;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import com.google.common.io.OutputSupplier;

public class Zips {

    public static void unzip(File zipFile, File destFolder) throws IOException {
        ZipInputStream zis = null;
        try {
            zis = new ZipInputStream(new FileInputStream(zipFile));
            ZipEntry entry = zis.getNextEntry();
            if (entry == null) {
                throw new IOException(MessageFormat.format("''{0}'' is empty or not a ZIP file.", zipFile));
            }
            do {
                if (!entry.isDirectory()) {
                    final File file = new File(destFolder, entry.getName());
                    Files.createParentDirs(file);
                    Files.asByteSink(file, FileWriteMode.APPEND).writeFrom(zis);
                }
            } while ((entry = zis.getNextEntry()) != null);
        } finally {
            Closeables.close(zis, true);
        }
    }

    public static void zip(File directory, File out) throws IOException {
        ZipOutputStream zos = null;
        try {
            OutputSupplier<FileOutputStream> s = Files.newOutputStreamSupplier(out);
            zos = new ZipOutputStream(s.getOutput());
            for (File f : FileUtils.listFiles(directory, FILE, DIRECTORY)) {
                String path = removeStart(f.getPath(), directory.getAbsolutePath() + File.separator);
                path = path.replace(File.separatorChar, '/');
                ZipEntry e = new ZipEntry(path);
                zos.putNextEntry(e);
                Files.asByteSource(f).copyTo(zos);
                zos.closeEntry();
            }
        } finally {
            Closeables.close(zos, false);
        }
    }

}
