/*******************************************************************************
 * Copyright (c) 1998, 2011 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the 
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0 
 * which accompanies this distribution. 
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at 
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
 *     05/16/2008-1.0M8 Guy Pelletier 
 *       - 218084: Implement metadata merging functionality between mapping files
 *     03/24/2011-2.3 Guy Pelletier 
 *       - 337323: Multi-tenant with shared schema support (part 1)
 ******************************************************************************/  
package org.eclipse.persistence.internal.jpa.metadata.queries;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.persistence.internal.jpa.EJBQueryImpl;
import org.eclipse.persistence.internal.jpa.metadata.accessors.MetadataAccessor;
import org.eclipse.persistence.internal.jpa.metadata.accessors.objects.MetadataAccessibleObject;
import org.eclipse.persistence.internal.jpa.metadata.accessors.objects.MetadataAnnotation;
import org.eclipse.persistence.internal.jpa.metadata.xml.XMLEntityMappings;
import org.eclipse.persistence.internal.sessions.AbstractSession;
import org.eclipse.persistence.queries.StoredProcedureCall;

/**
 * INTERNAL:
 * Object to hold onto a named stored procedure query.
 * 
 * Key notes:
 * - any metadata mapped from XML to this class must be compared in the
 *   equals method.
 * - all metadata mapped from XML should be initialized in the initXMLObject 
 *   method.
 * - when loading from annotations, the constructor accepts the metadata
 *   accessor this metadata was loaded from. Used it to look up any 
 *   'companion' annotation needed for processing.
 * - methods should be preserved in alphabetical order.
 * 
 * @author Guy Pelletier
 * @since TopLink 11g
 */
public class NamedStoredProcedureQueryMetadata extends NamedNativeQueryMetadata {
    private Boolean m_multipleResultSets;
    private Boolean m_returnsResultSet;
    private Boolean m_callByIndex;

    private List<StoredProcedureParameterMetadata> m_parameters = new ArrayList<StoredProcedureParameterMetadata>();
    
    private String m_procedureName;
    
    /**
     * INTERNAL:
     * Used for XML loading.
     */
    public NamedStoredProcedureQueryMetadata() {
        super("<named-stored-procedure-query>");
    }
    
    /**
     * INTERNAL:
     */
    public NamedStoredProcedureQueryMetadata(MetadataAnnotation namedStoredProcedureQuery, MetadataAccessor accessor) {
        super(namedStoredProcedureQuery, accessor);
         
        for (Object storedProcedureParameter : (Object[]) namedStoredProcedureQuery.getAttributeArray("parameters")) {
           m_parameters.add(new StoredProcedureParameterMetadata((MetadataAnnotation)storedProcedureParameter, accessor));
        }
        
        m_procedureName = (String) namedStoredProcedureQuery.getAttribute("procedureName");
        m_returnsResultSet = (Boolean) namedStoredProcedureQuery.getAttributeBooleanDefaultFalse("returnsResultSet");
        m_multipleResultSets = (Boolean) namedStoredProcedureQuery.getAttributeBooleanDefaultFalse("multipleResultSets");
        m_callByIndex = (Boolean) namedStoredProcedureQuery.getAttributeBooleanDefaultFalse("callByIndex");
    }
    
    /**
     * INTERNAL:
     * Used for XML loading.
     */
    public NamedStoredProcedureQueryMetadata(String elementName) {
        super(elementName);
    }
   
    /**
     * INTERNAL:
     */
    public boolean callByIndex() {
        return m_callByIndex == null ? false : m_callByIndex; 
    }
    
    /**
     * INTERNAL:
     */
    @Override
    public boolean equals(Object objectToCompare) {
        if (super.equals(objectToCompare) && objectToCompare instanceof NamedStoredProcedureQueryMetadata) {
            NamedStoredProcedureQueryMetadata query = (NamedStoredProcedureQueryMetadata) objectToCompare;
            
            if (! valuesMatch(m_returnsResultSet, query.getReturnsResultSet())) {
                return false;
            }
            
            if (! valuesMatch(m_callByIndex, query.getCallByIndex())) {
                return false;
            }
            
            if (! valuesMatch(m_multipleResultSets, query.getMultipleResultSets())) {
                return false;
            }
            
            if (! valuesMatch(m_parameters, query.getParameters())) {
                return false;
            }
            
            return valuesMatch(m_procedureName, query.getProcedureName());
        }
        
        return false;
    }

    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public Boolean getCallByIndex() {
        return m_callByIndex;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping
     */
    public Boolean getMultipleResultSets() {
        return m_multipleResultSets;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public List<StoredProcedureParameterMetadata> getParameters() {
        return m_parameters;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public String getProcedureName() {
        return m_procedureName;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping
     */
    public Boolean getReturnsResultSet() {
        return m_returnsResultSet;
    }
    
    /**
     * INTERNAL:
     */
    public boolean hasMultipleResultSets() {
        return m_multipleResultSets == null ? false : m_multipleResultSets; 
    }
    
    /**
     * INTERNAL:
     */
    @Override
    public void initXMLObject(MetadataAccessibleObject accessibleObject, XMLEntityMappings entityMappings) {
        super.initXMLObject(accessibleObject, entityMappings);
        
        // Initialize parameters ...
        initXMLObjects(m_parameters, accessibleObject);
    }
    
    /**
     * INTERNAL:
     */
    @Override
    public void process(AbstractSession session, ClassLoader loader) {
        // Build the stored procedure call.
        StoredProcedureCall call = new StoredProcedureCall();
        
        // Process the stored procedure parameters.
        boolean callByIndex = callByIndex();
        for (StoredProcedureParameterMetadata parameter : m_parameters) {
            parameter.process(call, callByIndex, false);
        }
        
        // Process the procedure name.
        call.setProcedureName(m_procedureName);
        
        // Process the returns result set.
        call.setReturnsResultSet(returnsResultSet());
        
        // Process the multiple result sets.
        call.setHasMultipleResultSets(hasMultipleResultSets());
        
        // Process the query hints.
        Map<String, Object> hints = processQueryHints(session);
        
        // Process the result class.
        if (getResultClass().isVoid()) {
            if (hasResultSetMapping(session)) {
                session.addQuery(getName(), EJBQueryImpl.buildStoredProcedureQuery(getResultSetMapping(), call, hints, loader, session));
            } else {
                // Neither a resultClass or resultSetMapping is specified so place in a temp query on the session
                session.addQuery(getName(), EJBQueryImpl.buildStoredProcedureQuery(call, hints, loader, session));
            }
        } else {
            session.addQuery(getName(), EJBQueryImpl.buildStoredProcedureQuery(MetadataHelper.getClassForName(getResultClass().getName(), loader), call, hints, loader, session));
        }
    }
    
    /**
     * INTERNAL:
     */
    public boolean returnsResultSet() {
        return m_returnsResultSet == null ? false : m_returnsResultSet; 
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public void setCallByIndex(Boolean callByIndex) {
        m_callByIndex = callByIndex;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping
     */
    public void setMultipleResultSets(Boolean multipleResultSets) {
        m_multipleResultSets = multipleResultSets;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public void setParameters(List<StoredProcedureParameterMetadata> parameters) {
        m_parameters = parameters;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public void setProcedureName(String procedureName) {
        m_procedureName = procedureName;
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public void setReturnsResultSet(Boolean returnsResultSet) {
        m_returnsResultSet = returnsResultSet;
    }
}
